/*
 * sys_pipe.c - open a pipe communication channel
 * 
 * Author    Alain Greiner  (2016,1017,2018,2019,2020)
 *
 * Copyright (c)  UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_kernel_types.h>
#include <vfs.h>
#include <hal_uspace.h>
#include <process.h>
#include <thread.h>
#include <printk.h>
#include <pipe.h>

#include <syscalls.h>

////////////////////////////
int sys_pipe ( fdid_t * fd )
{
    vseg_t       * vseg;
    pipe_t       * pipe;
    vfs_file_t   * file_0;
    vfs_file_t   * file_1;
    fdid_t         fdid_0;
    fdid_t         fdid_1;
    error_t        error;

    thread_t     * this    = CURRENT_THREAD;
    process_t    * process = this->process;

#if DEBUG_SYS_PIPE || DEBUG_SYSCALLS_ERROR || CONFIG_INSTRUMENTATION_SYSCALLS
uint64_t   tm_start = hal_get_cycles();
#endif

#if DEBUG_SYS_PIPE
if( DEBUG_SYS_PIPE < tm_end )
printk("\n[%s] thread[%x,%x] enter for <%s> / cycle %d\n",
__FUNCTION__, process->pid, this->trdid, pathname, (uint32_t)tm_end );
#endif

    // check user buffer is mapped 
    if( vmm_get_vseg( process , (intptr_t)fd , &vseg ) )
    {

#if DEBUG_SYSCALLS_ERROR
if( DEBUG_SYSCALLS_ERROR < (uint32_t)tm_start )
printk("\n[ERROR] in %s : thread[%x,%x] / output user buffer unmapped %x\n",
__FUNCTION__ , process->pid, this->trdid, (intptr_t)fd );
#endif
		this->errno = EINVAL;
		return -1;
    }

    // 1. allocate memory in local cluster for pipe descriptor,
    //    remote buf_descriptor, and associated data buffer
    pipe = pipe_create( local_cxy,
                        CONFIG_PIPE_BUF_SIZE );

    if( pipe == NULL )
    {

#if DEBUG_SYSCALLS_ERROR
if( DEBUG_SYSCALLS_ERROR < (uint32_t)tm_start )
printk("\n[ERROR] in %s : thread[%x,%x] / no memory for pipe\n",
__FUNCTION__ , process->pid, this->trdid );
#endif
        goto error_1;
    }

    // 2. allocate memory for fd[0] file descriptor in local cluster
    // we don't use the vfs_file_create function because there is no inode.
	file_0 = kmem_alloc( bits_log2(sizeof(vfs_file_t)) , AF_ZERO );

    if( file_0 == NULL ) 
    {

#if DEBUG_SYSCALLS_ERROR
if( DEBUG_SYSCALLS_ERROR < (uint32_t)tm_start )
printk("\n[ERROR] in %s : thread[%x,%x] / no memory for file descriptor\n",
__FUNCTION__, process->pid, this->trdid );
#endif
        goto error_2;
    }
    
    // 3. get fd[0] fdid value and register it in owner process fd_array[]
    error = process_fd_register( process->owner_xp,
                                 XPTR( local_cxy , file_0 ),
                                 &fdid_0 );
    if ( error )
    {

#if DEBUG_SYSCALLS_ERROR
if( DEBUG_SYSCALLS_ERROR < (uint32_t)tm_start )
printk("\n[ERROR] in %s : thread[%x,%x] / cannot register file descriptor \n",
__FUNCTION__, process->pid, this->trdid );
#endif
        goto error_3;
    }

    // 4. allocate memory for fd[1] file descriptor in local cluster
    // we don't use the vfs_file_create function because there is no inode.
	file_1 = kmem_alloc( bits_log2(sizeof(vfs_file_t)) , AF_ZERO );

    if( file_1 == NULL ) 
    {

#if DEBUG_SYSCALLS_ERROR
if( DEBUG_SYSCALLS_ERROR < (uint32_t)tm_start )
printk("\n[ERROR] in %s : thread[%x,%x] / no memory for file descriptor\n",
__FUNCTION__, process->pid, this->trdid );
#endif
        goto error_4;
    }

    // 5. get fd[1] fdid value and register it in owner process fd_array[]
    error = process_fd_register( process->owner_xp,
                                 XPTR( local_cxy , file_1 ),
                                 &fdid_1 );
    if ( error )
    {

#if DEBUG_SYSCALLS_ERROR
if( DEBUG_SYSCALLS_ERROR < (uint32_t)tm_start )
printk("\n[ERROR] in %s : thread[%x,%x] / cannot register file descriptor \n",
__FUNCTION__, process->pid, this->trdid );
#endif
        goto error_5;
    }

    // link the two file descriptors to the pipe
    file_0->pipe = pipe;
    file_1->pipe = pipe;

    // copy fdid_0 & fdid_1 values to user buffer
    hal_copy_to_uspace( &fd[0] , XPTR( local_cxy , &fdid_0 ) , sizeof(fdid_t) );
    hal_copy_to_uspace( &fd[1] , XPTR( local_cxy , &fdid_1 ) , sizeof(fdid_t) );

#if (DEBUG_SYS_PIPE || CONFIG_INSTRUMENTATION_SYSCALLS)
uint64_t     tm_end = hal_get_cycles();
#endif

#if DEBUG_SYS_PIPE
if( DEBUG_SYS_PIPE < tm_end )
printk("\n[%s] thread[%x,%x] exit for <%s> / cycle %d\n",
__FUNCTION__, this->process->pid, this->trdid, pathname, (uint32_t)tm_end );
#endif
 
#if CONFIG_INSTRUMENTATION_SYSCALLS
hal_atomic_add( &syscalls_cumul_cost[SYS_PIPE] , tm_end - tm_start );
hal_atomic_add( &syscalls_occurences[SYS_PIPE] , 1 );
#endif

    return 0;

error_5:    // release memory allocated for fd[1] file descriptor

    kmem_free( file_1 , bits_log2(sizeof(vfs_file_t)) );

error_4:    // release fdid_0 from fd_array[]

    process_fd_remove( process->ref_xp , fdid_0 );

error_3:    // release memory allocated for fd[0] file descriptor

    kmem_free( file_0 , bits_log2(sizeof(vfs_file_t)) );

error_2:    // release memory allocated for the pipe 

    pipe_destroy( XPTR( local_cxy , pipe ) );

error_1:   // set errno and return error

    this->errno = ENOMEM;
    return -1;

}  // end sys_pipe()
