/*
 * sys_fbf.c - Acces the frame buffer peripheral.
 * 
 * Authors     Alain Greiner (2016,2017,2018,2019,2020)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_kernel_types.h>
#include <hal_uspace.h>
#include <hal_vmm.h>
#include <shared_fbf.h>
#include <dev_fbf.h>
#include <errno.h>
#include <thread.h>
#include <printk.h>
#include <vmm.h>
#include <syscalls.h>

////////////////////////
int sys_fbf( reg_t arg0,
             reg_t arg1,
             reg_t arg2,
             reg_t arg3 )
{
	vseg_t         * vseg;        // for vaddr check
    error_t          error;

#if (DEBUG_SYS_FBF || CONFIG_INSTRUMENTATION_SYSCALLS)
uint64_t     tm_start = hal_get_cycles();
#endif

    thread_t       * this    = CURRENT_THREAD;
    process_t      * process = this->process;

    // for some operations, the MSB of arg0 can contain the wid 
    uint32_t operation = arg0 & 0xFFFF;

#if DEBUG_SYS_FBF
if( DEBUG_SYS_FBF < tm_start )
printk("\n[%s] thread[%x,%x] %s / a1 %x / a2 %x / a3 %x / cycle %d\n",
__FUNCTION__, process->pid, this->trdid, dev_fbf_cmd_str( operation ),
arg1, arg2, arg3, (uint32_t)tm_start );
#endif

    // execute requested operation
	switch( operation )
	{
        ////////////////////
	    case FBF_GET_CONFIG:
        {
            uint32_t width;
            uint32_t height;
            uint32_t type;

            // check "width" in user vspace
	        error = vmm_get_vseg( process , (intptr_t)arg1 , &vseg );
	        if( error )
            {

#if DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : unmapped arg1 %x for %s / thread[%x,%x]\n",
__FUNCTION__ , arg1, dev_fbf_cmd_str(operation), process->pid, this->trdid );
#endif
                this->errno = EINVAL;
            }

            // check "height" in user vspace
	        error |= vmm_get_vseg( process , (intptr_t)arg2 , &vseg );
	        if( error )
            {

#if DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : unmapped arg2 %x for %s / thread[%x,%x]\n",
__FUNCTION__ , arg2, dev_fbf_cmd_str(operation), process->pid, this->trdid );
#endif
                this->errno = EINVAL;
            }

            // check "type" in user vspace
	        error |= vmm_get_vseg( process , (intptr_t)arg3 , &vseg );
	        if( error )
            {

#if DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : unmapped arg3 %x for %s / thread[%x,%x]\n",
__FUNCTION__ , arg3, dev_fbf_cmd_str(operation), process->pid, this->trdid );
#endif
                this->errno = EINVAL;
            }

            // call relevant kernel function to get  config
            dev_fbf_get_config( &width , &height , &type );

            // transfer to user space if no error
            if( error == 0 )
            {
                hal_copy_to_uspace( (void*)arg1, XPTR(local_cxy , &width ), sizeof(uint32_t) );
                hal_copy_to_uspace( (void*)arg2, XPTR(local_cxy , &height), sizeof(uint32_t) );
                hal_copy_to_uspace( (void*)arg3, XPTR(local_cxy , &type  ), sizeof(uint32_t) );
            }
            break;
        }
        /////////////////////
        case FBF_DIRECT_READ:
        case FBF_DIRECT_WRITE:
        {
            void    * buffer   = (void *)arg1;
            uint32_t  npixels  = arg2;
            uint32_t  offset   = arg3;
            bool_t    is_write = (operation == FBF_DIRECT_WRITE);
            
            // check buffer in user space
	        error = vmm_get_vseg( process , (intptr_t)buffer , &vseg );
	        if( error )
            {

#if DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : unmapped buffer %x for %s / thread[%x,%x]\n",
__FUNCTION__ , buffer, dev_fbf_cmd_str(operation), process->pid, this->trdid );
#endif
                this->errno = EINVAL;
            }
            else
            {
                // call relevant kernel function
                error |= dev_fbf_move_data( is_write, buffer, npixels, offset );           

                if( error )
                {

#if DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : cannot move data for %s / buffer %x / thread[%x,%x]\n",
__FUNCTION__ , dev_fbf_cmd_str(operation), buffer, process->pid, this->trdid );
#endif
                    this->errno = EINVAL;
                }
            }
            break;
        }
        ///////////////////////
        case FBF_CREATE_WINDOW:
        {
            uint32_t   l_zero  = arg1 >> 16;
            uint32_t   p_zero  = arg1 & 0xFFFF;
            uint32_t   nlines  = arg2 >> 16;
            uint32_t   npixels = arg2 & 0xFFFF;

            // check buffer in user space
	        error = vmm_get_vseg( process , (intptr_t)arg3 , &vseg );
	        if( error )
            {

#if DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : unmapped user buffer %x for %s / thread[%x,%x]\n",
__FUNCTION__ , (intptr_t)arg3, dev_fbf_cmd_str(operation), process->pid, this->trdid );
#endif
                this->errno = EINVAL;
            }
            else
            {
                // allocated buffer base address
                intptr_t  user_buffer;
     
                // call relevant kernel function
                error = dev_fbf_create_window( nlines,
                                               npixels,
                                               l_zero,
                                               p_zero,
                                               &user_buffer );
                if( error == -1 )
                {

#if DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : cannot create window for %s / thread[%x,%x]\n",
__FUNCTION__ , dev_fbf_cmd_str(operation), process->pid, this->trdid );
#endif
                    this->errno = EINVAL;
                }

                // copy vseg base address to user space buffer
                hal_copy_to_uspace( (void *)arg3,
                                    XPTR( local_cxy , &user_buffer ),
                                    sizeof(intptr_t) );
                hal_fence();
            }
            break;
        }
        ///////////////////////
        case FBF_ACTIVE_WINDOW:
        {
            uint32_t  wid    = arg1;
            uint32_t  active = arg2;

            // call relevant kernel function
            error = dev_fbf_active_window( wid , active );
                
            if( error )
            {

#if DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : cannot %s / thread[%x,%x]\n",
__FUNCTION__ , dev_fbf_cmd_str(operation), process->pid, this->trdid );
#endif
                this->errno = EINVAL;
            }
            break;
        }
        ///////////////////////
        case FBF_DELETE_WINDOW:
        {
            uint32_t  wid = arg1;

            // call relevant kernel function
            error = dev_fbf_delete_window( wid );
                
            if( error )
            {

#if DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : cannot %s / thread[%x,%x]\n",
__FUNCTION__ , dev_fbf_cmd_str(operation), process->pid, this->trdid );
#endif
                this->errno = EINVAL;
            }
            break;
        }
        ////////////////////////
        case FBF_REFRESH_WINDOW:
        {
            uint32_t  wid        = arg1;
            uint32_t  line_first = arg2;
            uint32_t  line_last  = arg3;

            // call relevant kernel function
            error = dev_fbf_refresh_window( wid,
                                            line_first,
                                            line_last );
                
            if( error )
            {

#if DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : cannot %s / thread[%x,%x]\n",
__FUNCTION__ , dev_fbf_cmd_str(operation), process->pid, this->trdid );
#endif
                this->errno = EINVAL;
            }
            break;
        }
        /////////////////////
        case FBF_MOVE_WINDOW:
        {
            uint32_t  wid      = arg1; 
            uint32_t  l_zero   = arg2;
            uint32_t  p_zero   = arg3;

            // call relevant kernel function to move data
            error = dev_fbf_move_window( wid, l_zero, p_zero ); 

            if( error )
            {

#if DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : cannot %s / thread[%x,%x]\n",
__FUNCTION__ , dev_fbf_cmd_str(operation), process->pid, this->trdid );
#endif
                this->errno = EINVAL;
            }
            break;
        }
        ///////////////////////
        case FBF_RESIZE_WINDOW:
        {
            uint32_t  wid      = arg1; 
            uint32_t  width    = arg2;
            uint32_t  height   = arg3;

            // call relevant kernel function to move data
            error = dev_fbf_resize_window( wid , width , height ); 

            if( error )
            {

#if DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : cannot %s / thread[%x,%x]\n",
__FUNCTION__ , dev_fbf_cmd_str(operation), process->pid, this->trdid );
#endif
                this->errno = EINVAL;
            }
            break;
        }
        //////////////////////
        case FBF_FRONT_WINDOW:
        {
            uint32_t  wid = arg1;

            // call relevant kernel function
            error = dev_fbf_front_window( wid );
                
            if( error )
            {

#if DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : cannot %s / thread[%x,%x]\n",
__FUNCTION__ , dev_fbf_cmd_str(operation), process->pid, this->trdid );
#endif
                this->errno = EINVAL;
            }
            break;
        }
        ///////
	    default:  // undefined operation                       
        {

#if DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : undefined operation type %d / thread %x in process %x / cycle %d\n",
__FUNCTION__ , operation, this->trdid, process->pid, (uint32_t)hal_get_cycles() );
#endif
            this->errno = EINVAL;
            error = -1;
        }
        break;
	}  // end switch on operation

    hal_fence();

#if (DEBUG_SYS_FBF || CONFIG_INSTRUMENTATION_SYSCALLS)
uint64_t     tm_end = hal_get_cycles();
#endif

#if DEBUG_SYS_FBF
if( DEBUG_SYS_FBF < tm_end )
printk("\n[%s] thread[%x,%x] exit for %s / cycle %d\n",
__FUNCTION__, process->pid, this->trdid, dev_fbf_cmd_str( operation ), (uint32_t)tm_end );
#endif

#if CONFIG_INSTRUMENTATION_SYSCALLS
hal_atomic_add( &syscalls_cumul_cost[SYS_FBF] , tm_end - tm_start );
hal_atomic_add( &syscalls_occurences[SYS_FBF] , 1 );
#endif

    return error;

}  // end sys_fbf()
