/*
 * sys_trace.c - activate / desactivate the context switches trace for a given core
 * 
 * Author    Alain Greiner (c) (2016,2017,2018,2019)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_kernel_types.h>
#include <hal_special.h>
#include <printk.h>
#include <thread.h>
#include <errno.h>
#include <shared_syscalls.h>
#include <syscalls.h>

///////////////////////////////
int sys_trace( bool_t   active,
               cxy_t    cxy,
               lid_t    lid )
{
    uint32_t    ncores;

    thread_t  * this    = CURRENT_THREAD;

#if DEBUG_SYS_TRACE || DEBUG_SYSCALLS_ERROR || CONFIG_INSTRUMENTATION_SYSCALLS
process_t * process  = this->process;
uint64_t    tm_start = hal_get_cycles();
#endif

#if DEBUG_SYS_TRACE
if( DEBUG_SYS_TRACE < tm_start )
printk("\n[%s] thread[%x,%x] enters / cycle = %d\n",
__FUNCTION__, this->process->pid, this->trdid, (uint32_t)tm_start );
#endif

    // check cluster identifier
    if( cluster_is_active( cxy ) == false )
    {

#if DEBUG_SYSCALLS_ERROR
if( DEBUG_SYSCALLS_ERROR < (uint32_t)tm_start )
printk("\n[ERROR] in %s : thread[%x,%x] / illegal cxy argument %x\n",
__FUNCTION__ , this->trdid , process->pid , cxy );
#endif
        this->errno = EINVAL;
        return -1;
    }

    // check core local index
    ncores = hal_remote_l32( XPTR( cxy , &LOCAL_CLUSTER->cores_nr ) );
    if( lid >= ncores )
    {

#if DEBUG_SYSCALLS_ERROR
if( DEBUG_SYSCALLS_ERROR < (uint32_t)tm_start )
printk("\n[ERROR] in %s : thread[%x,%x] / illegal lid argument %x\n",
__FUNCTION__ , this->trdid , process->pid , lid );
#endif
        this->errno = EINVAL;
        return -1;
    }

    // get local pointer on target core
    core_t * core = &LOCAL_CLUSTER->core_tbl[lid];

    // get extended pointer on target scheduler trace field
    xptr_t trace_xp = XPTR( cxy , &core->scheduler.trace );

    if ( active ) hal_remote_s32( trace_xp , 1 );
    else          hal_remote_s32( trace_xp , 0 );
    
    hal_fence();

#if (DEBUG_SYS_TRACE || CONFIG_INSTRUMENTATION_SYSCALLS)
uint64_t     tm_end = hal_get_cycles();
#endif

#if DEBUG_SYS_TRACE
if( DEBUG_SYS_TRACE < tm_end )
printk("\n[%s] thread[%x,%x] exit / cycle %d\n",
__FUNCTION__, this->process->pid, this->trdid, (uint32_t)tm_end );
#endif

#if CONFIG_INSTRUMENTATION_SYSCALLS
hal_atomic_add( &syscalls_cumul_cost[SYS_TRACE] , tm_end - tm_start );
hal_atomic_add( &syscalls_occurences[SYS_TRACE] , 1 );
#endif

    return 0;

}  // end sys_trace()
