/*
 * hal_context.h - Generic Thread Context Access API definition.
 * 
 * Author  Alain Greiner    (2016,2017,2018,2019,2020)
 *
 * Copyright (c) UPMC Sorbonne Universites
 * 
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef  _HAL_CONTEXT_H_
#define  _HAL_CONTEXT_H_

//////////////////////////////////////////////////////////////////////////////////////////
//        Generic Thread Context API definition (implementation in hal_context.c) 
//
// A thread context is defined by the two (core specific) structures hal_cpu_context_t
// and hal_fpu_context_t, defined in hal_context.c file, that are accessed with generic
// void* pointers stored in the thread descriptor.
// - the "hal_cpu_context_t" struct saves the CPU registers values at context switch.
// - the "hal_fpu_context_t" struct saves the FPU registers values at FPU switch.
//////////////////////////////////////////////////////////////////////////////////////////

/**** Forward declarations ****/

struct thread_s;

/****************************************************************************************
 * This function allocates memory for a CPU context and links it to the thread
 * identified by the <thread> argument. The context is not initialised.
 ****************************************************************************************
 * @ thread  : pointer on the thread descriptor.
 * @ return 0 if success / return -1 if failure.
 ***************************************************************************************/
error_t hal_cpu_context_alloc( struct thread_s * thread );

/****************************************************************************************
 * This function initializes the CPU context of the thread identified by the <thread>
 * argument from the <thread> descriptor, depending on the thread type (user / kernel).
 * All slots required to start a new thread must be initialized in the thread context.
 * Moreover, for an user thread, the arguments to be passed to the entry function 
 * depend on the <is_main> argument below:
 * - a main thread, created by sys_exec(), requires two arguments, defined by the two
 *   <argc> and <argv> arguments below.
 *   define the arguments 
 * - a pthread, created by sys_thread_create() requires only one argument, defined by
 *   the "entry_args" field in the thread descriptor.
 ****************************************************************************************
 * @ thread  : pointer on the thread descriptor.
 * @ is_main : main thread when non zero.
 * @ argc    : actual number of arguments (for main).
 * @ argv    : user space pointer on array of pointers on strings arguments (for main).
 ***************************************************************************************/
void hal_cpu_context_init( struct thread_s * thread,
                           bool_t            is_main,
                           uint32_t          argc,
                           intptr_t          argv );

/****************************************************************************************
 * This function is called the sys_fork() function to complete the fork mechanism. 
 * It is called by th local parent thread to initialize the CPU context of the remote
 * child thread, identified by the <thread_xp> argument.
 * It makes three actions:
 * 1) It copies the current values of the CPU registers of the core running the parent
 *    thread to the remote child CPU context.
 * 2) It patches four slots of this remote child CPU context:
 *    - the c0_th   slot is set to the child thread descriptor pointer. 
 *    - the sp_29   slot is set to the child kernel stack pointer.
 *    - the c0_sr   slot is set to kernel mode with IRQ disabled.
 *    - the c2_ptpr slot is set to the child process GPT value.
 * 3) It copies the content of the parent thread kernel_stack, to the child thread 
 *    kernel_stack, because the COW mechanism is not available on architectures where
 *    the data MMU is de-activated in kernel mode.
 ****************************************************************************************
 * @ thread_xp  : extended pointer on the child thread descriptor.
 ***************************************************************************************/
void hal_cpu_context_fork( xptr_t    thread_xp );

/****************************************************************************************
 * This function display some slots of the CPU context.
 ****************************************************************************************
 * @ thread_xp  : extended pointer on the thread descriptor.
 ***************************************************************************************/
void hal_cpu_context_display( xptr_t  thread_xp );

/****************************************************************************************
 * This function releases the physical memory allocated for a thread CPU context.
 ****************************************************************************************
 * @ thread  : pointer on the thread descriptor.
 ***************************************************************************************/
void hal_cpu_context_destroy( struct thread_s * thread );







/****************************************************************************************
 * This function allocates memory for a FPU context, reset all entries,
 * and links it to the thread identified by the <thread> argument.
 ****************************************************************************************
 * @ thread  : pointer on the thread descriptor.
 * @ return 0 if success / return -1 if failure.
 ***************************************************************************************/
error_t hal_fpu_context_alloc( struct thread_s * thread );

/****************************************************************************************
 * This function initializes a FPU context from scratch.
 ****************************************************************************************
 * @ thread  : pointer on the thread descriptor.
 ***************************************************************************************/
void hal_fpu_context_init( struct thread_s * thread );

/****************************************************************************************
 * This function copies a FPU context defined by the <src> argument to the FPU context
 * defined by the <dst> argument. It is used by the fork system call.
 ****************************************************************************************
 * @ dst  : pointer on the destination thread descriptor.
 * @ src  : pointer on the source thread descriptor.
 ***************************************************************************************/
void hal_fpu_context_copy( struct thread_s * dst,
                           struct thread_s * src );

/****************************************************************************************
 * This function releases the physical memory allocated for a FPU context.
 ****************************************************************************************
 * @ thread  : pointer on the thread descriptor.
 ***************************************************************************************/
void hal_fpu_context_destroy( struct thread_s * thread );

/****************************************************************************************
 * This function is used to implement the fork() system call. 
 * It saves in a remote thread FPU context the current FPU registers values.
 ****************************************************************************************
 * @ thread_xp  : extended pointer on the remote thread descriptor.
 ***************************************************************************************/
void hal_fpu_context_save( xptr_t thread_xp );

/****************************************************************************************
 * This function restores from the calling thread FPU context the FPU registers values.
 ****************************************************************************************
 * @ thread  : pointer on the thread descriptor.
 ***************************************************************************************/
void hal_fpu_context_restore( struct thread_s * thread );

#endif	/* _HAL_CONTEXT_H_ */
