/*
 * dev_mmc.c - MMC (Memory Cache Controler) generic device API implementation.
 * 
 * Author  Alain Greiner    (2016)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MK
 *
 * ALMOS-MKH.is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH.is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-kernel; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_types.h>
#include <hal_special.h>
#include <soclib_mmc.h>
#include <printk.h>
#include <chdev.h>
#include <thread.h>
#include <dev_mmc.h>

/////////////////////////////////////////////////////////////////////////////////////////
// Extern global variables
/////////////////////////////////////////////////////////////////////////////////////////

extern chdev_directory_t  chdev_dir;         // allocated in kernel_init.c

extern chdev_icu_input_t  chdev_icu_input;   // allocated in kernel_init.c

////////////////////////////////////
void dev_mmc_init( chdev_t * chdev )
{
    // get implementation from device descriptor
    uint32_t  impl = chdev->impl;

    // set driver specific fields in device descriptor and call driver init function
    if( impl == IMPL_MMC_TSR )
    {
        chdev->cmd = &soclib_mmc_cmd;
        chdev->isr = &soclib_mmc_isr;
        soclib_mmc_init( chdev );
    }
    else
    {
        assert( false , __FUNCTION__ , "undefined MMC device implementation" );
    }
}  // end dev_mmc_init()

/////////////////////////////////////////////////////////////////////////////
// This static function makes some checking, takes the lock granting
// exclusive access to MMC peripheral, call the driver to execute the command
// registered in the calling thread descriptor, and releases the lock.
/////////////////////////////////////////////////////////////////////////////
static void dev_mmc_access( thread_t * this )
{
    mmc_dmsg("\n[INFO] %s enters for thread %x in process %x : command = %d\n", 
                 __FUNCTION__ , this->trdid , this->process->pid , this->command.mmc.type );

    // get extended pointer on MMC device
    xptr_t  dev_xp  = this->command.mmc.dev_xp;

    assert( (dev_xp != XPTR_NULL) , __FUNCTION__ , "undefined MMC device descriptor" );

    // get MMC device cluster identifier & local pointer
    cxy_t     dev_cxy = GET_CXY( dev_xp );
    chdev_t * dev_ptr = (chdev_t *)GET_PTR( dev_xp );

    // get driver command function pointer from MMC device descriptor
    dev_cmd_t * cmd = (dev_cmd_t *)hal_remote_lpt( XPTR( dev_cxy , &dev_ptr->cmd ) );

    // get the MMC device remote spinlock
    remote_spinlock_lock( XPTR( dev_cxy , &dev_ptr->wait_lock ) );  

    // call driver command
    cmd( XPTR( local_cxy , this ) );
    
    // release the MMC device remote spinlock
    remote_spinlock_unlock( XPTR( dev_cxy , &dev_ptr->wait_lock ) );  

    mmc_dmsg("\n[INFO] %s completes for thread %x in process %x / error = %d\n", 
                 __FUNCTION__ , this->trdid , this->process->pid , this->dev.ioc.error );

}  // end dev_mmc_access()

////////////////////////////////////////////
error_t dev_mmc_inval( paddr_t    buf_paddr,
                       uint32_t   buf_size )
{
    // get calling thread local pointer
    thread_t * this = CURRENT_THREAD;

    // get target cluster from paddr
    cxy_t cxy = CXY_FROM_PADDR( buf_paddr );

    assert( ((buf_paddr & (CONFIG_CACHE_LINE_SIZE -1)) == 0) , __FUNCTION__ ,
             "buffer not aligned on cache line" );

    // get extended pointer on MMC device descriptor
    xptr_t  dev_xp = chdev_dir.mmc[cxy];

    // store command arguments in thread descriptor
    this->command.mmc.dev_xp    = dev_xp;
    this->command.mmc.type      = MMC_CC_INVAL;
    this->command.mmc.buf_paddr = buf_paddr;
    this->command.mmc.buf_size  = buf_size;

    // execute operation
    dev_mmc_access( this ); 

    // return operation status
    return this->command.mmc.error;  
}  

/////////////////////////////////////
error_t dev_mmc_sync( paddr_t    buf_paddr,
                      uint32_t   buf_size )
{
    // get calling thread local pointer
    thread_t * this = CURRENT_THREAD;

    // get target cluster from paddr
    cxy_t cxy = CXY_FROM_PADDR( buf_paddr );

    assert( ((buf_paddr & (CONFIG_CACHE_LINE_SIZE -1)) == 0) , __FUNCTION__ ,
             "buffer not aligned on cache line" );

    // store command arguments in thread descriptor
    this->command.mmc.dev_xp    = chdev_dir.mmc[cxy];
    this->command.mmc.type      = MMC_CC_SYNC;
    this->command.mmc.buf_paddr = buf_paddr;
    this->command.mmc.buf_size  = buf_size;

    // execute operation
    dev_mmc_access( this ); 

    // return operation status
    return this->command.mmc.error;  
}  

/////////////////////////////////////////
error_t dev_mmc_set_error( cxy_t     cxy,
                           uint32_t  index,
                           uint32_t  wdata )
{
    // get calling thread local pointer
    thread_t * this = CURRENT_THREAD;

    // store command arguments in thread descriptor
    this->command.mmc.dev_xp    = chdev_dir.mmc[cxy];
    this->command.mmc.type      = MMC_SET_ERROR;
    this->command.mmc.reg_index = index;
    this->command.mmc.reg_ptr   = &wdata;

    // execute operation
    dev_mmc_access( this ); 

    // return operation status
    return this->command.mmc.error;  
}
                        
//////////////////////////////////////////
error_t dev_mmc_get_error( cxy_t      cxy,
                           uint32_t   index,
                           uint32_t * rdata )
{
    // get calling thread local pointer
    thread_t * this = CURRENT_THREAD;

    // store command arguments in thread descriptor
    this->command.mmc.dev_xp    = chdev_dir.mmc[cxy];
    this->command.mmc.type      = MMC_GET_ERROR;
    this->command.mmc.reg_index = index;
    this->command.mmc.reg_ptr   = rdata;

    // execute operation
    dev_mmc_access( this ); 

    // return operation status
    return this->command.mmc.error;  
}
                        
////////////////////////////////////////////////////
error_t dev_mmc_get_instrumentation( cxy_t      cxy,
                                     uint32_t   index,
                                     uint32_t * rdata )
{
    // get calling thread local pointer
    thread_t * this = CURRENT_THREAD;

    // store command arguments in thread descriptor
    this->command.mmc.dev_xp    = chdev_dir.mmc[cxy];
    this->command.mmc.type      = MMC_GET_INSTRU;
    this->command.mmc.reg_index = index;
    this->command.mmc.reg_ptr   = rdata;

    // execute operation
    dev_mmc_access( this ); 

    // return operation status
    return this->command.mmc.error;  
}

