/*
 * remote_fifo.h - Lock-less Single-Reader Multiple-Writers FIFO
 *
 * Authors : Mohamed Lamine Karaoui (2015)
 *           Alain Greiner (2016,2017)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MHH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef _REMOTE_FIFO_H_
#define _REMOTE_FIFO_H_

#include <kernel_config.h>
#include <hal_types.h>
#include <errno.h>
#include <hal_remote.h>

/************************************************************************************
 * This structure defines a generic, single reader, multiple writers
 * remote FIFO, that is used by the RPCs for inter cluster communications.
 * The accesses are implemented using a lock-free algorithm, as it uses a ticket 
 * based mechanism to handle concurrent access between multiple writers.
 * Each FIF0 slot can contain one 64 bits integer.
 * In case of FIFO full, the writer deschedule without blocking, to retry later.
 *
 * WARNING : the number of slots is statically defined by the global 
 * configuration parameter CONFIG_REMOTE_FIFO_SLOTS for all fifos, requiring
 * 12 * CONFIG_REMOTE_FIFO_SLOTS bytes for each FIFO.
 ***********************************************************************************/

typedef struct remote_fifo_s
{
	volatile uint32_t  wr_id;                            /*! write slot index      */
	volatile uint32_t  rd_id;                            /*! read  slot index      */
    volatile uint32_t  valid[CONFIG_REMOTE_FIFO_SLOTS];  /*! empty slot if 0       */
	uint64_t           data[CONFIG_REMOTE_FIFO_SLOTS];   /*! fifo slot content     */
}
remote_fifo_t;

/************************************************************************************
 * This function initializes the local FIFO as empty.
 * It can only initialise a local FIFO.
 ************************************************************************************
 * @ fifo    : pointer to the local fifo.
 ***********************************************************************************/
void local_fifo_init( remote_fifo_t * fifo );

/************************************************************************************
 * This non blocking function tries to get one item from the local fifo.
 * The reader must get exclusive access before calling this function.  
 * The read slot index is incremented.
 ************************************************************************************
 * @ fifo    : pointer to the local fifo.
 * @ item    : pointer on destination buffer for extracted item.
 * @ size    : actual number of bytes in one item.
 * @ return  0 on success, EAGAIN if the buffer is empty.
 ***********************************************************************************/
error_t local_fifo_get_item( remote_fifo_t * fifo, 
                             uint64_t      * item );

/************************************************************************************
 * This blocking function puts one item to a remote fifo identified
 * by an extended pointer.
 * This function gets a write ticket using a remote_atomic_increment on the
 * write slot index and waits until the slot is empty, using a descheduling
 * policy (without blocking).
 ************************************************************************************
 * @ fifo    : extended pointer to the fifo in remote cluster.
 * @ item    : pointer on a local buffer containing the item to be stored.
 * @ first   : [out] true if first item registered in remote fifo.
 * @ return  0 on success / EBUSY if a contention has been detected.
 ***********************************************************************************/
error_t remote_fifo_put_item( xptr_t     fifo,
                              uint64_t * item,
                              bool_t   * first );

/************************************************************************************
 * Query if local fifo is empty
 ************************************************************************************
 * @ fifo    : pointer to the fifo.
 * @ return true if the fifo is empty, false otherwise.
 ***********************************************************************************/
bool_t local_fifo_is_empty( remote_fifo_t * fifo );

/************************************************************************************
 * Query if remote fifo is full
 ************************************************************************************
 * @ fifo    : pointer to the fifo in remote cluster.
 * @ cxy	 : remote cluster index.
 * @ return true if the fifo is full, false otherwise.
 ***********************************************************************************/
bool_t remote_fifo_is_full( xptr_t fifo );

/************************************************************************************
 * Query number ot items in remote fifo.
 ************************************************************************************
 * @ fifo     : pointer to the fifo in remote cluster.
 * @ cxy	  : remote cluster index.
 * @ return  number of items.
 ***********************************************************************************/
uint32_t remote_fifo_items( xptr_t fifo );


#endif	/* _REMOTE_FIFO_H_ */
