/*
 * soclib_pic.c - soclib PIC driver implementation.
 *
 * Author  Alain Greiner (2016)
 * *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH. is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH. is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH.; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_types.h>
#include <chdev.h>
#include <soclib_pic.h>
#include <soclib_xcu.h>
#include <errno.h>
#include <string.h>
#include <vfs.h>


////////////////////////////////////////
void soclib_pic_init( chdev_t  * chdev )
{
    // get PIC controller segment cluster and local pointer
    cxy_t      seg_cxy = (cxy_t)GET_CXY( chdev->base );
    uint32_t * seg_ptr = (uint32_t *)GET_PTR( chdev->base );
    uint32_t   i;

    // reset the MASK registers for all input IRQs
    for( i = 0 ; i < CONFIG_MAX_IRQS_PER_PIC ; i++ )
    {
        hal_remote_sw( XPTR( seg_cxy , seg_ptr + i*IOPIC_SPAN + IOPIC_MASK ) , 0 ); 
    }
}

////////////////////////////////////////////
void soclib_pic_bind_irq( xptr_t     dev_xp,
                          uint32_t   irq_id,
                          xptr_t     xp_wti )
{
    // get PIC device descriptor cluster and local pointer
    cxy_t     dev_cxy = GET_CXY( dev_xp );
    chdev_t * dev_ptr = (chdev_t *)GET_PTR( dev_xp );
  
    // get extended pointer on PIC segment base from PIC device descriptor
    xptr_t seg_xp = (xptr_t)hal_remote_lwd( XPTR( dev_cxy , &dev_ptr->base ) );
 
    // get PIC controller segment cluster and local pointer
    cxy_t      seg_cxy = (cxy_t)GET_CXY( seg_xp);
    uint32_t * seg_ptr = (uint32_t *)GET_PTR( seg_xp );

    uint32_t lsb = (uint32_t)xp_wti;
    uint32_t msb  = (uint32_t)(xp_wti>>32);

	// set the IOPIC_ADDRESS and IOPIC_EXTEND registers
    hal_remote_sw( XPTR( seg_cxy , seg_ptr+irq_id*IOPIC_SPAN+IOPIC_ADDRESS ) , lsb );
    hal_remote_sw( XPTR( seg_cxy , seg_ptr+irq_id*IOPIC_SPAN+IOPIC_EXTEND  ) , msb );

    // set IOPIC_MASK register
    hal_remote_sw( XPTR( seg_cxy , seg_ptr+irq_id*IOPIC_SPAN+IOPIC_MASK    ) , 1   );
}

//////////////////////////////////////////////
void soclib_pic_unbind_irq( xptr_t     dev_xp,
                            uint32_t   irq_id )
{
    // get PIC device descriptor cluster and local pointer
    cxy_t     dev_cxy = GET_CXY( dev_xp );
    chdev_t * dev_ptr = (chdev_t *)GET_PTR( dev_xp );
  
    // get extended pointer on PIC segment base from PIC device descriptor
    xptr_t seg_xp = (xptr_t)hal_remote_lwd( XPTR( dev_cxy , &dev_ptr->base ) );
 
    // get PIC controller segment cluster and local pointer
    cxy_t      seg_cxy = (cxy_t)GET_CXY( seg_xp);
    uint32_t * seg_ptr = (uint32_t *)GET_PTR( seg_xp );

    // set IOPIC_MASK register
    hal_remote_sw( XPTR( seg_cxy , seg_ptr+irq_id*IOPIC_SPAN+IOPIC_MASK    ) , 0   );
}

//////////////////////////////////////////////
void soclib_pic_get_status( xptr_t     dev_xp,
                            uint32_t   irq_id,
                            uint32_t * status)
{
    // get PIC device descriptor cluster and local pointer
    cxy_t     dev_cxy = GET_CXY( dev_xp );
    chdev_t * dev_ptr = (chdev_t *)GET_PTR( dev_xp );
  
    // get extended pointer on PIC segment base from PIC device descriptor
    xptr_t seg_xp = (xptr_t)hal_remote_lwd( XPTR( dev_cxy , &dev_ptr->base ) );
 
    // get PIC controller segment cluster and local pointer
    cxy_t      seg_cxy = (cxy_t)GET_CXY( seg_xp);
    uint32_t * seg_ptr = (uint32_t *)GET_PTR( seg_xp );

    // return status
	*status = hal_remote_lw( XPTR( seg_cxy , seg_ptr+irq_id*IOPIC_SPAN+IOPIC_STATUS ) );
}

