/*
 * dev_pic.h - PIC (External Interrupt Controler) generic device API definition.
 *
 * Authors   Alain Greiner  (2016)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef _DEV_PIC_H_
#define _DEV_PIC_H_

#include <kernel_config.h>
#include <hal_types.h>

/*****************************************************************************************
 *     Generic External Interrupt Controler definition
 *
 * The PIC generic device describes an external interrupt Controler, that translates 
 * N input IRQs generated by external peripherals to N WTI IRQs, that will be routed 
 * to a dynamically allocated WTI mailbox in a given cluster. 
 *
 * The max number of input IRQs is defined by the CONFIG_MAX_IRQS_PER_PIC parameter. 
 * The actual number of connected IRQs is defined in the ''arch_info'' file, and stored 
 * in the PIC device extension.  
 * The "source" device for each input IRQ is also defined in the ''arch_info'' file,
 * and are stored in the ''chdev_input_irq'' global variable in kernel initialization.
 *
 * This external peripheral does not execute I/O operations, but is just acting as a 
 * dynamically configurable interrupt router for another I/O operation. Therefore, 
 * ALMOS-MKH does not use the PIC device waiting queue, but call directly the relevant 
 * driver blocking functions to dynamically "bind" or "unbind" an external IRQ to a
 * WTI mailbox.
 ****************************************************************************************/
 
/****  Forward declarations  ****/

struct chdev_s;

/*****************************************************************************************
 * This defines the (implementation independant) extension for the PIC device.
 ****************************************************************************************/

typedef struct pic_extend_s
{
    uint32_t    irq_nr;     /*! actual number of input IRQs to the PIC device           */
}
pic_extend_t;

/*****************************************************************************************
 * This enum defines the various implementations of the PIC external interrupt controller.
 * This array must be kept consistent with the define in arch_info.h file
 ****************************************************************************************/

enum pic_impl_e
{
    IMPL_PIC_SOC =   0,     
    IMPL_PIC_I86 =   1,
}
pic_impl_t;

/*****************************************************************************************
 * This function makes two initialisations:
 * - It initializes the PIC specific fields of the chdev descriptor.
 * - it initializes the implementation specific PIC hardware device.
 * It must be executed once in the kernel initialisation phase.
 *****************************************************************************************
 * @ xp_dev     : extended pointer on PIC device descriptor.
 * @ uint32_t   : actual number of input IRQs.
 ****************************************************************************************/
void dev_pic_init( struct chdev_s * chdev,
                   uint32_t         irq_nr );

/*****************************************************************************************
 * This function link a WTI mailbox to the PIC input IRQ identified by its index,
 * and unmask the selected input IRQ, calling the relevant driver.
 *****************************************************************************************
 * @ irq_id    : input IRQ index.
 * @ cxy       : WTI mailbox cluster.
 * @ wti_id    : WTI mailbox index in cluster.
 ****************************************************************************************/
void dev_pic_bind_irq( uint32_t   irq_id,
                       cxy_t      cxy,
                       uint32_t   wti_id );

/*****************************************************************************************
 * This function mask a PIC input IRQ identified by its index, calling te relevant driver.
 *****************************************************************************************
 * @ irq_id    : input IRQ index.
 ****************************************************************************************/
void dev_pic_unbind_irq( uint32_t   irq_id );
                        

#endif  /* _DEV_PIC_H_ */
