/*
 * xhtab.h - Remote access embedded hash table definition.
 * 
 * Author     Alain Greiner  (2016,2017)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef _XHTAB_H_
#define _XHTAB_H_

#include <kernel_config.h>
#include <hal_types.h>
#include <remote_rwlock.h>
#include <xlist.h>


///////////////////////////////////////////////////////////////////////////////////////////
// This file define a generic, embedded, remotely accessible hash table.
//
// It can be accessed by any thread, running in any cluster.
// It is generic as it can be used to register various types of items.
// The main goal is to speedup search by key for a large number of items of same type.
// For this purpose the set of all registered items is split in several subsets.
// Each subset is organised an embedded double linked lists.
// - an item is uniquely identified by a <key>, that can be a single uint32_t, 
//   a name (character string), or a more complex structure.
// - From the pointer on <key>, we use an item type specific xhtab_index() function,
//   to compute an <index> value, defining a subset of registered items.
// - to discriminate between items that have the same <index>, the hash table uses another
//   item type specific "xhtab_scan()" function for the associative search in subset.
// - Each registered item is a structure, that must contain an embedded xlist_entry,
//   that is part of the xlist implementing the subset. 
//
// Implementation Note: for each supported item type ***, you must define the two
//                      xhtab_***_index() and xhtab_***_scan() functions, and
//                      update the xhtab_init() function.
///////////////////////////////////////////////////////////////////////////////////////////

#define HASHTAB_SIZE    64   // number of subsets

/******************************************************************************************
 * These typedef define the two item type specific function prototypes.
 *****************************************************************************************/

typedef  xptr_t    xhtab_scan_t( xptr_t xhtab_xp , uint32_t index , void * key );

typedef  uint32_t  xhtab_index_t( void * key );

/******************************************************************************************
 * This define the supported item types.
 *****************************************************************************************/

typedef enum
{
    XHTAB_DENTRY_TYPE = 0,                    /*! item is a vfs_dentry_t                 */ 
}
xhtab_item_type_t;

/******************************************************************************************
 * This structure define the root of the remote accessible hash table. 
 *****************************************************************************************/

typedef struct xhtab_s
{
	xlist_entry_t      roots[HASHTAB_SIZE];   /*! array of roots of xlist                */
    xhtab_index_t    * index;                 /*! item specific function                 */
    xhtab_scan_t     * scan;                  /*! item specific function                 */
    uint32_t           items;                 /*! number of registered items             */
    remote_rwlock_t    lock;                  /*! lock protecting hash table accesses    */
}
xhtab_t;

/******************************************************************************************
 * This function initializes an empty hash table (zero registered item).
 * The initialisation must be done by a thread running in cluster containing the table.
 ******************************************************************************************
 * @ xhtab   : local pointer on local xhtab to be initialized.
 * @ type    : item type (see above).
 *****************************************************************************************/
void xhtab_init( xhtab_t           * xhtab,
                 xhtab_item_type_t   type );

/******************************************************************************************
 * This function safely register an item in the hash table, using the lock protecting it.
 ******************************************************************************************
 * @ xhtab_xp   : extended pointer on hash table.
 * @ key        : local pointer on item identifier.
 * @ xlist_xp   : extended pointer on xlist_entry_t embedded in item to be registered.
 * @ return 0 if success / return EINVAL if item already registered.
 *****************************************************************************************/
error_t xhtab_insert( xptr_t   xhtab_xp,
                      void   * key,
                      xptr_t   xlist_xp );

/******************************************************************************************
 * This function safely remove an item from the hash table, using the lock protecting it.
 ******************************************************************************************
 * @ xhtab_xp       : extended pointer on hash table.
 * @ key            : local pointer on item identifier.
 * @ xlist_entry_xp : extended pointer on xlist_entry embedded in item to be removed.
 * @ return 0 if success / return EINVAL if item not found.
 *****************************************************************************************/
error_t xhtab_remove( xptr_t   xhtab_xp,
                      void   * key,
                      xptr_t   xlist_entry_xp );

/******************************************************************************************
 * This function search an item by its key in hash table, using the lock protecting it.
 ******************************************************************************************
 * @ xhtab_xp  : extended pointer on hash table.
 * @ key       : local pointer on searched item identifier.
 * @ return extended pointer on the searched item if found / XPTR_NULL if not found.
 *****************************************************************************************/
xptr_t  xhtab_lookup( xptr_t    xhtab_xp,
                      void    * key );


#endif	/* _XHTAB_H_ */
