/*
 * soclib_xcu.c - soclib XCU driver API implementation.
 *
 * Authors  Alain Greiner    (2016)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH.is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH.is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-kernel; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <soclib_xcu.h>
#include <hal_types.h>
#include <core.h>
#include <chdev.h>



////////////////////////////////////
void soclib_xcu_init( chdev_t * icu,
                      lid_t     lid )
{
    // get local ICU segment base address
	uint32_t * base = (uint32_t *)GET_PTR( icu->base );

    // disable all IRQs
    base[XCU_MSK_HWI_DISABLE << 5 | lid] = 0xFFFFFFFF;
    base[XCU_MSK_WTI_DISABLE << 5 | lid] = 0xFFFFFFFF;
    base[XCU_MSK_PTI_DISABLE << 5 | lid] = 0xFFFFFFFF;
}

////////////////////////////////////////////
void soclib_xcu_disable_irq( chdev_t  * icu,
                             uint32_t   mask,
                             uint32_t   type,
                             lid_t      lid )
{
    // get XCU segment base address
	uint32_t * base = (uint32_t *)GET_PTR( icu->base );

    // write into register
	if     ( type == WTI_TYPE ) base[XCU_MSK_WTI_DISABLE << 5 | lid] = mask;
	else if( type == HWI_TYPE ) base[XCU_MSK_HWI_DISABLE << 5 | lid] = mask;
	else                        base[XCU_MSK_PTI_DISABLE << 5 | lid] = mask;
}

///////////////////////////////////////////
void soclib_xcu_enable_irq( chdev_t  * icu,
                            uint32_t   mask,
                            uint32_t   type,
                            lid_t      lid )
{
    // get XCU segment base address
	uint32_t * base = (uint32_t *)GET_PTR( icu->base );

    // write into register
	if     ( type == WTI_TYPE ) base[XCU_MSK_WTI_ENABLE << 5 | lid] = mask;
	else if( type == HWI_TYPE ) base[XCU_MSK_HWI_ENABLE << 5 | lid] = mask;
	else                        base[XCU_MSK_PTI_ENABLE << 5 | lid] = mask;
}

///////////////////////////////////////////
void soclib_xcu_get_masks( chdev_t   * icu,
                           lid_t       lid,
                           uint32_t  * hwi_mask,
                           uint32_t  * wti_mask,
                           uint32_t  * pti_mask )
{
    // get XCU segment base address
	uint32_t * base = (uint32_t *)GET_PTR( icu->base );
 
    // get values from registers
    *hwi_mask = base[XCU_MSK_HWI << 5 | lid]; 
    *wti_mask = base[XCU_MSK_WTI << 5 | lid]; 
    *pti_mask = base[XCU_MSK_PTI << 5 | lid]; 
}

//////////////////////////////////////////
void soclib_xcu_set_period( chdev_t * icu,
                            uint32_t  index,
                            uint32_t  period )
{
    // get local ICU segment base address
	uint32_t * base = (uint32_t *)GET_PTR( icu->base );

    // write into register
    base[XCU_PTI_PER << 5 | index] = period;
}

/////////////////////////////////////////////
uint32_t soclib_xcu_ack_timer( chdev_t * icu,
                               uint32_t  index )
{
    // get local ICU segment base address
	uint32_t * base = (uint32_t *)GET_PTR( icu->base );

    // read from register
	return base[XCU_PTI_ACK << 5 | index];
}

///////////////////////////////////////////
void soclib_xcu_get_status( chdev_t  * icu,
                            lid_t      lid,
                            uint32_t * hwi_status,
                            uint32_t * wti_status,
                            uint32_t * pti_status )
{
    // get local ICU segment base address 
    uint32_t * base = (uint32_t *)GET_PTR( icu->base );

    // read PRIO register
	uint32_t prio = base[XCU_PRIO << 5 | lid];

    *wti_status = (prio & 0x4) ? (((prio >> 24) & 0x1F) + 1) : 0;
    *hwi_status = (prio & 0x2) ? (((prio >> 16) & 0x1F) + 1) : 0;
    *pti_status = (prio & 0x1) ? (((prio >>  8) & 0x1F) + 1) : 0;
}

/////////////////////////////////////////
void soclib_xcu_send_ipi( xptr_t  icu_xp,
                          lid_t   lid )
{
    // get target ICU device cluster and local pointer
    cxy_t       cxy_icu = GET_CXY( icu_xp );
    chdev_t  * ptr_icu = (chdev_t *)GET_PTR( icu_xp );

    // get extended pointer on target ICU segment base
    xptr_t  xp_base = (xptr_t)hal_remote_lwd( XPTR( cxy_icu , &ptr_icu->base ) );

    // get remote ICU segment local pointer 
	uint32_t  * base = (uint32_t *)GET_PTR( xp_base );

    // send IPI to remote core
	hal_remote_sw( XPTR( cxy_icu , &base[XCU_WTI_REG << 5 | lid] ) , 0 );
}

//////////////////////////////////////////////
uint32_t * soclib_xcu_wti_ptr( chdev_t  * icu,
                               uint32_t   index )
{
    uint32_t * base = (uint32_t *)GET_PTR( icu->base );

    return &base[XCU_WTI_REG << 5 | index];
}
