/*
 * hal_apic.c - Advanced Programmable Interrupt Controller
 *
 * Copyright (c) 2017 Maxime Villard
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH.; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_types.h>
#include <hal_boot.h>
#include <hal_register.h>
#include <hal_segmentation.h>
#include <hal_apic.h>
#include <hal_internal.h>

#include <memcpy.h>
#include <thread.h>
#include <string.h>
#include <process.h>
#include <printk.h>
#include <vmm.h>
#include <core.h>
#include <cluster.h>

/* -------------------------------------------------------------------------- */

#define PIC1_CMD	0x0020
#define PIC1_DATA	0x0021
#define PIC2_CMD	0x00a0
#define PIC2_DATA	0x00a1

static void hal_pic_init()
{
	/*
	 * Disable the PIC (8259A). We are going to use IOAPIC instead.
	 */
	out8(PIC1_DATA, 0xff);
	out8(PIC2_DATA, 0xff);
}

/* -------------------------------------------------------------------------- */

paddr_t lapic_pa __in_kdata = 0;
vaddr_t lapic_va __in_kdata = 0;

void hal_lapic_write(uint32_t reg, uint32_t val)
{
	*((volatile uint32_t *)((uint8_t *)lapic_va + reg)) = val;
}

uint32_t hal_lapic_read(uint32_t reg)
{
	return *((volatile uint32_t *)((uint8_t *)lapic_va + reg));
}

uint32_t hal_lapic_gid()
{
	return hal_lapic_read(LAPIC_ID) >> LAPIC_ID_SHIFT;
}

/*
 * We have 8 interrupt sources:
 *  - Spurious
 *  - APIC Timer (TMR)
 *  - Local Interrupt 0 (LINT0)
 *  - Local Interrupt 1 (LINT1)
 *  - Performance Monitor Counters (PMC)
 *  - Thermal Sensors (THM)
 *  - APIC internal error (ERR)
 *  - Extended (Implementation dependent)
 */
static void hal_lapic_init()
{
	lapic_va = hal_gpt_bootstrap_valloc(1); // XXX: should be shared

	if ((rdmsr(MSR_APICBASE) & APICBASE_PHYSADDR) != lapic_pa) {
		x86_panic("APICBASE and ACPI don't match!\n");
	}
	wrmsr(MSR_APICBASE, lapic_pa | APICBASE_EN);

	hal_gpt_enter(lapic_va, lapic_pa, PG_V|PG_KW|PG_NX|PG_N);

	hal_lapic_write(LAPIC_TPR, 0);
	hal_lapic_write(LAPIC_EOI, 0);
	hal_lapic_write(LAPIC_SVR, LAPIC_SVR_ENABLE|LAPIC_SPURIOUS_VECTOR);

	/* Explicitly disable (mask) each vector */
	hal_lapic_write(LAPIC_LVT_TMR, LAPIC_TMR_M);
	hal_lapic_write(LAPIC_LVT_LINT0, LAPIC_LINT_M);
	hal_lapic_write(LAPIC_LVT_LINT1, LAPIC_LINT_M);
	hal_lapic_write(LAPIC_LVT_PMC, LAPIC_PMC_M);
	hal_lapic_write(LAPIC_LVT_THM, LAPIC_THM_M);
	hal_lapic_write(LAPIC_LVT_ERR, LAPIC_ERR_M);
}

/* -------------------------------------------------------------------------- */

void hal_apic_init()
{
	/* Disable the PIC */
	hal_pic_init();

	/* Enable the LAPIC */
	hal_lapic_init();
}

