/*
 * devfs.h - DEVFS File System API definition                                
 *
 * Authors       Mohamed Lamine Karaoui (2014,2015)
 *               Alain Greiner (2016,2017)
 *
 * Copyright (c) 2011,2012 UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef _DEVFS_H_
#define _DEVFS_H_

///////////////////////////////////////////////////////////////////////////////////////////
// The DEVFS File System contains inodes associated to all chdev descriptors availables
// in the target architecture.  It is a three levels tree structure:
// - The "dev" directory inode is stored in cluster_0. It is the root of the devfs
//   file system. The parent inode is the "/" global root.
// - There is one "internal_cxy" directory inode per cluster, that is the parent of
//   the local children inodes associated to the local chdev descriptors. 
//   The parent inode and dentry are stored in cluster_0.
// - The I/O cluster contains one "external" directory inode, that is the parent of 
//   the remote children inodes associated to the remote external descriptors.
//   The parent inode and dentry are stored in cluster_0.
//
// The DEVFS File system does not require a context extension, but requires an inode
// extension to store the pointer on the associated chdev descriptor.
///////////////////////////////////////////////////////////////////////////////////////////

/**************  Forward declarations     ************************************************/

struct chdev_s;

/******************************************************************************************
 * This structure defines the DEVFS inode descriptor extension.
 *****************************************************************************************/

typedef struct devfs_inode_ext_s
{
    struct chdev_s * chdev;       /* extended pointer on associated chdev                */
}
devfs_inode_t;





//////////////////////////////////////////////////////////////////////////////////////////
//                 These functions are called by the VFS.
//////////////////////////////////////////////////////////////////////////////////////////

/******************************************************************************************
 * This function  does not exist for the DEVFS file system, as this File System
 * cannot be mounted as the root FS.
 *****************************************************************************************/
xptr_t devfs_init();


/******************************************************************************************
 * This function mount the DEVFS file system on the root FS.
 * It is executed cooperatively during kernel init by all CP0s in all clusters, 
 * to create the DEVFS 3-levels tree structure. 
 * This is a four phases procedure, and it uses the global barrier to synchronize 
 * the cooperating kernel instances when required.
 * - phase 1 : In all clusters, it creates the local extension of the DEVFS context,
 *    and initialises it.
 * - phase 2 : In cluster_0 only, it creates the dentry and the associated inode for
 *   the DEVFS root directory. 
 * - phase 3 : In all cluster(x,y), it creates the dentry and the associated inode for 
 *   the "internal_cxy" directory. It also creates the dentry and associated inode
 *   for all local internal chdevs.
 * - phase 4 : In cluster_io only, it creates the dentry and the associated inode
 *   for the "external" directory. It also creates the dentry and associated inode
 *   for all external chdevs.
 ******************************************************************************************
 * @ parent_inode_xp   : extended pointer on the parent inode in root FS.
 * @ devfs_root_name   : DEVFS root directory name.
 *****************************************************************************************/
error_t devfs_mount( xptr_t   parent_inode_xp,
                     char   * devfs_root_name );

/*****************************************************************************************
 * This function initializes all fields of the VFS context.
 * No extra memory is allocated for a DEVFS context.
 *****************************************************************************************
 * @ vfs_ctx        : local pointer on VFS context for FATFS.
 * @ root_inode_xp  : extended pointer on the VFS root inode.
 ****************************************************************************************/
error_t devfs_ctx_init( struct vfs_ctx_s * vfs_ctx,
                        xptr_t             root_inode_xp );

/*****************************************************************************************
 * This function does not exist for a DEVFS context, as there is no DEVFS context.
 ****************************************************************************************/
error_t devfs_ctx_destroy();

/*****************************************************************************************
 * This function allocates memory for a DEVFS inode, initialise it,
 * and link it to the VFS inode.
 *****************************************************************************************
 * @ inode          : local pointer on the VFS inode.
 * @ chdev          : local pointer on the chdev to be inserted in DEVFS.
 ****************************************************************************************/
error_t devfs_inode_create( struct vfs_inode_s * inode,
                            struct chdev_s     * chdev );

/*****************************************************************************************
 * This function releases memory allocated for a DEVFS inode.
 *****************************************************************************************
 * @ inode   : local pointer on vfs_inode.
 ****************************************************************************************/
void devfs_inode_destroy( struct vfs_inode_s * inode );

/*****************************************************************************************
 * This function does not exist for the DEVFS File System.
 ****************************************************************************************/
error_t devfs_write_page( struct page_s * page );

/*****************************************************************************************
 * This function does not exist for the DEVFS File System.
 ****************************************************************************************/
error_t devfs_read_page( struct page_s * page );



#endif	/* _DEVFS_H_ */
