/*
 * dev_txt.c - TXT (Text Terminal) generic device API implementation.
 * 
 * Author  Alain Greiner    (2016)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MK
 *
 * ALMOS-MKH.is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH.is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-kernel; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_types.h>
#include <hal_special.h>
#include <hal_remote.h>
#include <soclib_tty.h>
#include <thread.h>
#include <rpc.h>
#include <printk.h>
#include <dev_txt.h>

/////////////////////////////////////////////////////////////////////////////////////////
// Extern global variables
/////////////////////////////////////////////////////////////////////////////////////////

extern chdev_directory_t  chdev_dir;         // allocated in kernel_init.c

extern chdev_pic_input_t  chdev_pic_input;   // allocated in kernel_init.c

////////////////////////////////////
void dev_txt_init( chdev_t * chdev )
{
    // the local ICU chdev must be initialized before the TXT chdev, because
    // the TXT chdev initialisation requires allocation of a WTI from local ICU
    xptr_t  icu_xp  = chdev_dir.icu[local_cxy];
    assert( (icu_xp != XPTR_NULL) , __FUNCTION__ , "ICU not initialised before TXT" );

    // get implementation and channel index
    uint32_t  impl    = chdev->impl;
    uint32_t  channel = chdev->channel;

    // set fields "cmd", "isr", and call driver init function
    if( impl == IMPL_TXT_TTY )
    {
        chdev->cmd = &soclib_tty_cmd;
        chdev->isr = &soclib_tty_isr;
        soclib_tty_init( chdev );
    }
    else
    {
        assert( false , __FUNCTION__ , "undefined TXT device implementation" );
    }

    // get a WTI mailbox from local ICU
    uint32_t wti_id = dev_icu_wti_alloc();

    assert( (wti_id != -1) , __FUNCTION__ , "cannot allocate WTI mailbox" );

    // select a core
    lid_t lid = cluster_select_local_core();

    // enable WTI IRQ and update WTI interrupt vector
    dev_icu_enable_irq( lid , WTI_TYPE , wti_id , chdev );

    // link IOC IRQ to WTI mailbox in PIC component 
    uint32_t irq_id = chdev_pic_input.txt[channel];
    dev_pic_bind_irq( irq_id , local_cxy , wti_id );

    // create server thread
    thread_t * new_thread;
    error_t    error;

    error = thread_kernel_create( &new_thread,
                                  THREAD_DEV,
                                  &chdev_sequencial_server,
                                  chdev,
                                  lid ); 
    assert( (error == 0) , __FUNCTION__ , "cannot create server thread" );

    // set "server" field in chdev descriptor
    chdev->server = new_thread;
    
    // start server thread
    thread_unblock( XPTR( local_cxy , new_thread ) , THREAD_BLOCKED_GLOBAL );

} // end dev_txt_init()


//////////////////////////////////////////////////////////////////////////////////
// This static function is called by dev_txt_read(), dev_txt_write() functions.
////////////////////////////////////i/////////////////////////////////////////////
static error_t dev_txt_access( uint32_t   type,
                               uint32_t   channel,
                               char     * buffer,
                               uint32_t   count )
{
    thread_t * this = CURRENT_THREAD;

    txt_dmsg("\n[INFO] in %s : thread %x in process %x enters\n", 
                 __FUNCTION__ , this->trdid , this->process->pid );

    // check channel argument
    assert( (channel < CONFIG_MAX_TXT_CHANNELS) , __FUNCTION__ , "illegal channel index" );

    // get extended pointer on remote TXT chdev descriptor
    xptr_t  dev_xp = chdev_dir.txt[channel];

    assert( (dev_xp != XPTR_NULL) , __FUNCTION__ , "undefined TXT chdev descriptor" );

    // register command in calling thread descriptor
    this->command.txt.dev_xp  = dev_xp;
    this->command.txt.type    = type;
    this->command.txt.buf_xp  = XPTR( local_cxy , buffer );
    this->command.txt.count   = count;

    // register client thread in waiting queue, activate server thread
    // block client thread on THREAD_BLOCKED_IO and deschedule.
    // it is re-activated by the ISR signaling IO operation completion.
    chdev_register_command( dev_xp , this );

    txt_dmsg("\n[INFO] in %s : thread %x in process %x completes / error = %d\n", 
             __FUNCTION__ , this->trdid , this->process->pid , this->command.txt.error );

    // return I/O operation status from calling thread descriptor
    return this->command.txt.error;  

}  // end dev_txt_access()

/////////////////////////////////////////
error_t dev_txt_write( uint32_t   channel,
                       char     * buffer,
                       uint32_t   count )
{
    return dev_txt_access( TXT_WRITE , channel , buffer , count );
}
 
/////////////////////////////////////////
error_t dev_txt_read( uint32_t   channel,
                      char     * buffer )
{
    return dev_txt_access( TXT_READ , channel , buffer , 1 );
}

///////////////////////////////////////////////
error_t dev_txt_sync_write( uint32_t   channel,
                            char     * buffer,
                            uint32_t   count )
{
    // get pointer on calling thread 
    thread_t * this = CURRENT_THREAD;

    // get extended pointer on TXT[0] chdev 
    xptr_t  dev_xp = chdev_dir.txt[channel];

    // register command in calling thread 
    this->command.txt.dev_xp  = dev_xp;
    this->command.txt.type    = TXT_SYNC_WRITE;
    this->command.txt.buf_xp  = XPTR( local_cxy , buffer );
    this->command.txt.count   = count;

    // get driver command function 
    cxy_t       dev_cxy = GET_CXY( dev_xp );
    chdev_t   * dev_ptr = (chdev_t *)GET_PTR( dev_xp );
    dev_cmd_t * cmd = (dev_cmd_t *)hal_remote_lpt( XPTR( dev_cxy , &dev_ptr->cmd ) );

    // call directly driver command after taking chdev lock
    remote_spinlock_lock( XPTR( dev_cxy , &dev_ptr->wait_lock ) );
    cmd( XPTR( local_cxy , this ) );
    remote_spinlock_unlock( XPTR( dev_cxy , &dev_ptr->wait_lock ) );

    // return I/O operation status from calling thread descriptor
    return this->command.txt.error;
  
}  // end dev_txt_sync_write()

