/*
 * dev_ioc.h - IOC (Block Device Controler) generic device API definition.
 * 
 * Author  Alain Greiner    (2016)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-kernel; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef _DEV_IOC_H
#define _DEV_IOC_H

#include <hal_types.h>

/****  Forward declarations  ****/

struct device_s;

/*****************************************************************************************
 *     Generic Block Device Controler definition
 *
 * This device provide access to an external mass storage peripheral such as a
 * magnetic hard disk or a SD card, that can store blocks of data in a linear array
 * of sectors indexed by a simple lba (logic block address).
 * It supports two command types:
 * - READ  : move a given number of contiguous blocks from device to a memory buffer.
 * - WRITE : move a given number of contiguous blocks from a memory buffer to device.
 *
 * An I/O operation requires dynamic ressource allocation, and is always blocking for
 * the client thread. The general scenario is detailed below.
 * A) the client thread start the I/O operation, by calling the dev_ioc_read()
 *    or the dev_ioc_write() kernel functions that perform the following actions: 
 *    1) it get a free WTI mailbox from the client cluster WTI allocator.
 *    2) it enables the WTI IRQ on the client cluster ICU and update interrupt vector.
 *    3) it access the PIC to link the WTI mailbox to the IOC IRQ.
 *    4) it builds the command descriptor.
 *    5) it registers in the IOCdevice waiting queue. 
 *    6) itblock on the THREAD_BLOCKED_IO condition and deschedule.
 * B) The server thread attached to the IOC device descriptor handles the commands
 *    registered in the waiting queue, calling the IOC driver function.
 *    Most hardware implementation have a DMA capability, but some implementations,
 *    such as the RDK (Ram Disk) implementation does not use DMA.
 * C) The ISR signaling the I/O operation completion reactivates the client thread,
 *    that releases the allocated resources:
 *    1) access the PIC to unlink the IOC IRQ.
 *    2) disable the WTI IRQ in the client cluster ICU and update interrupt vector.
 *    3) release the WTI mailbox to the client cluster WTI allocator.
 *****************************************************************************************/

/******************************************************************************************
 * This defines the (implementation independant) extension for the generic IOC device.
 *****************************************************************************************/

typedef struct ioc_extend_s
{
    uint32_t    size;      /*! number of bytes in a block                               */
    uint32_t    count;     /*! total number of blocks in physical device                */
}
ioc_extend_t;

/******************************************************************************************
 * This enum defines the various implementations of the generic IOC peripheral.
 * It must be kept consistent with the define in arch_info.h file.
 *****************************************************************************************/

enum ioc_impl_e
{
    IMPL_IOC_BDV =   0,     
    IMPL_IOC_HBA =   1,  
    IMPL_IOC_SDC =   2,
    IMPL_IOC_SPI =   3,
    IMPL_IOC_RDK =   4,
}
ioc_impl_t;

/******************************************************************************************
 * This defines the (implementation independant)  command passed to the driver.
 *****************************************************************************************/

typedef struct ioc_command_s
{
    xptr_t      dev_xp;      /*! extended pointer on device descriptor                    */
    uint32_t    to_mem;     /*! requested operation (WRITE if zero / READ if non-zero)   */
    uint32_t    lba;        /*! first block index                                        */
    uint32_t    count;      /*! number of blocks                                         */
    xptr_t      buf_xp;     /*! extended pointer on memory buffer                        */
    uint32_t    error;      /*! operation status (0 if success)                          */
}
ioc_command_t;

/******************************************************************************************
 * This function completes the IOC device descriptor initialisation,
 * namely the link with the implementation specific driver.
 * The func, impl, channel, is_rxt, base, and size fields must be previously initialised.
 * It calls the specific driver initialisation function, to initialise the hardware
 * device and the specific data structures when required.
 * It creates the associated server thread.
 * It can be executed in another cluster than the cluster containing the device descriptor
 * or the hardware device itself.
 ******************************************************************************************
 * @ xp_dev     : extended pointer on IOC device descriptor.
 *****************************************************************************************/
void dev_ioc_init( xptr_t  xp_dev );

/******************************************************************************************
 * This blocking function try to tranfer one or several contiguous blocks of data
 * from the block device to a memory buffer. The corresponding request is actually
 * registered in the device pending request queue, and the calling thread is descheduled,
 * waiting on transfer completion. It will be resumed by the IRQ signaling completion.
 * It must be called in the client cluster.
 ******************************************************************************************
 * @ buffer    : local pointer on target buffer in memory.
 * @ lba       : first block index on device.
 * @ count     : number of blocks to transfer.
 * @ returns 0 if success / returns EINVAL if error.
 *****************************************************************************************/
error_t dev_ioc_read( char         * buffer,
                      uint32_t       lba,
                      uint32_t       count );

/******************************************************************************************
 * This blocking function try to tranfer one or several contiguous blocks of data
 * from a memory buffer to the block device. The corresponding request is actually
 * registered in the device pending request queue, and the calling thread is descheduled,
 * waiting on transfer completion. It will be resumed by the IRQ signaling completion.
 * It must be called in the client cluster.
 ******************************************************************************************
 * @ buffer    : local pointer on source buffer in memory.
 * @ lba       : first block index on device.
 * @ count     : number of blocks to transfer.
 * @ returns 0 if success / returns EINVAL if error.
 *****************************************************************************************/
error_t dev_ioc_write( char         * buffer,
                       uint32_t       lba,
                       uint32_t       count );

/******************************************************************************************
 * This function is executed by the server thread associated to the IOC device descriptor. 
 * This thread is created and activated by the dev_ioc_init() function.
 * It executes an infinite loop to handle sequencially all commands registered 
 * by the client threads in the device waiting queue, calling the driver CMD function. 
 *  
 * - If the peripheral can only handle one single command, the driver block the server 
 *   thread on the THREAD_BLOCKED_DEV_ISR condition, waiting I/O operation conmpletion.
 *   The server thread must be reacticated by the driver ISR function.
 * - If the peripheral can handle several commands in parallel (AHCI is an example), the
 *   driver does not block the server thread (it is only descheduled if the number of
 *   commands exceeeds the max number of parallel commands supported by the peripheral.
 *
 * When the waiting queue is empty, the server thread blocks on the THREAD_BLOCKED_CMD
 * condition and deschedule. It is re-activated by a client thread registering a command.
 ******************************************************************************************
 * @ dev     : local pointer on IOC device descriptor.
 *****************************************************************************************/
void dev_ioc_server( struct device_s * dev );

#endif	/* _DEV_IOC_H */
