/*
 * dev_txt.h - TXT (Text Terminal) generic device API definition.
 * 
 * Author  Alain Greiner    (2016)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-kernel; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef _DEV_TXT_H_
#define _DEV_TXT_H_

#include <almos_config.h>
#include <hal_types.h>

/****  Forward declarations  ****/

struct device_s ;

/******************************************************************************************
 *     Generic Text Terminal device definition.
 *
 * This multi-channels generic TXT device provide access to a text terminal.
 * It supports three operation types :
 * - TXT_READ : read a single character from the text terminal identified by its channel
 *   index, using a descheduling strategy for the calling thread.
 * - TXT_WRITE : write a character string to the text terminal identified by its channel
 *   index, using a descheduling strategy for the calling thread.
 * - TXT_SYNC_WRITE : write a character string to the terminal identified by its channel
 *   index, using a busy waiting strategy for the calling thread. 
 *****************************************************************************************/

/******************************************************************************************
 * This enum defines the various implementations of the generic TXT peripheral.
 * This array must be kept consistent with the define in arch_info.h file
 *****************************************************************************************/

enum txt_impl_e
{
    IMPL_TXT_TTY =   0,     
    IMPL_TXT_X86 =   1,  
}
txt_impl_t;

/******************************************************************************************
 * This defines the (implementation independant) command passed to the driver.
 *****************************************************************************************/

enum
{
    TXT_READ       = 0,
    TXT_WRITE      = 1,
    TXT_SYNC_WRITE = 2,
};

typedef struct txt_command_s
{
    xptr_t      dev_xp;    /*! extended pointer on the relevant TXT device descriptor    */
    uint32_t    type;      /*! TXT_READ / TXT_WRITE / TXT_SYNC_WRITE                     */
    xptr_t      buf_xp;    /*! extended pointer on characters array                      */
    uint32_t    count;     /*! number of characters in buffer (must be 1 if to_mem)      */
    uint32_t    error;     /*! operation status (0 if success)                           */
}
txt_command_t;

/******************************************************************************************
 * This function completes the TXT device descriptor initialisation,
 * namely the link with the implementation specific driver.
 * The func, impl, channel, is_rxt, base, and size fields must be previously initialised.
 * It calls the specific driver initialisation function, to initialise the hardware
 * device and the driver specific data structures when required.
 * It creates the associated server thread.
 * It can be executed in another cluster than cluster containing the device descriptor
 * or the hardware device itself.
 ******************************************************************************************
 * @ xp_dev     : extended pointer on TXT device descriptor.
 *****************************************************************************************/
void dev_txt_init( xptr_t xp_dev );

/******************************************************************************************
 * This blocking function read a single character from the terminal identified
 * by the "channel" argument. The corresponding request is actually registered in the
 * device pending requests queue, and the calling thread is descheduled, blocked until
 * transfer completion.
 * It must be called in the client cluster.
 ******************************************************************************************
 * @ channel   : TXT channel index.
 * @ buffer    : local pointer on destination buffer for the character.
 * @ returns 0 if success / returns EINVAL if error.
 *****************************************************************************************/
error_t dev_txt_read( uint32_t        channel,
                      char          * buffer );

/******************************************************************************************
 * This blocking function writes a character string on the terminal identified
 * by the "channel" argument. The corresponding request is actually registered in the
 * device pending requests queue, and the calling thread is descheduled, blocked until
 * transfer completion.
 * It must be called in the client cluster.
 ******************************************************************************************
 * @ channel   : TXT channel index.
 * @ buffer    : local pointer on source buffer containing the string.
 * @ count     : number of characters.
 * @ returns 0 if success / returns EINVAL if error.
 ****************************************************************************************/
error_t dev_txt_write( uint32_t        channel,
                       char          * buffer,
                       uint32_t        count );

 /***************************************************************************************
 * This low-level blocking function is used by the kernel to display one character on a 
 * given TXT channel without descheduling the calling thread,  without registering it 
 * in the TXT device waiting queue, and without using the TXT irq. 
 ****************************************************************************************
 * @ channel   : TXT channel index.
 * @ buffer    : local pointer on source buffer containing the string.
 * @ count     : number of characters.
 * @ returns 0 if success / returns EINVAL if error.
 ***************************************************************************************/
error_t dev_txt_sync_write( uint32_t   channel,
                            char     * buffer,
                            uint32_t   count );

/******************************************************************************************
 * This function is executed by the server thread associated to the TXT device descriptor.
 * This thread is created and lauched by the dev_txt_init() function, and is used
 * by the TXT_READ and TXT_WRITE operation types. It executes an infinite loop to handle
 * sequencially all commands registered by the client threads in the device waiting queue.
 *  
 * The driver CMD function is blocking, and returns only when the command is completed.
 * can use a busy waiting policy, or block the server thread on the THREAD_BLOCKED_IO_ISR
 * condition and deschedule, using the ISR to reactivate the server thread.
 *
 * When the waiting queue is empty, the server thread blocks on the THREAD_BLOCKED_IO_CMD
 * condition and deschedule. It is re-activated by a client thread registering a command.
 ******************************************************************************************
 * @ dev     : local pointer on device descriptor.
 *****************************************************************************************/
void dev_txt_server( struct device_s * dev );


#endif	/* _DEV_TXT_H_ */
