/*
 * sys_sem.c - Acces a POSIX unamed semaphore.
 * 
 * Copyright (c) 2008,2009,2010,2011,2012 Ghassan Almaless
 * Copyright (c) 2011,2012 UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-kernel.
 *
 * ALMOS-kernel is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-kernel is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-kernel; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_types.h>
#include <hal_uspace.
#include <remote_sem.h>

//////////////////////////////////
int sys_sem( void         * vaddr,        // semaphore virtual  address 
             uint32_t       operation,    // requested operation type
             uint32_t     * value )       // pointer on in/out argument
{
	uint32_t             data;   
	error_t              error;

    thread_t           * this      = CURRENT_THREAD;
	process_t          * process   = CURRENT_PROCESS;

    // check vaddr in user vspace
	error = vmm_check_address( process , vaddr , sizeof(unsigned long) );
	if( error ) 
    {
        this->errno = error;
        return -1;
    }

    // check value in user vspace
    error = vmm_check_address( process , value , sizeof(int*) );
	if( error ) 
    {
        this->errno = error;
        return -1;   
    }
    
    // execute requested operation
	switch( operation )
	{
        //////////////
	    case SEM_INIT:
        {
            // get argument
		    hal_copy_from_uspace( &data , value , sizeof(uint32_t) );

            // call init function
            error = remote_sem_init( (intptr_t)vaddr , data );

            if ( error )
            {
                this->errno = error;
                return -1;
            }
            break;
        }
        //////////////////
        case SEM_GETVALUE:
        {
            // get extended pointer on remote semaphore
            xptr_t sem_xp = remote_sem_from_vaddr( (intptr_t)vaddr );

            if( sem_xp == XPTR_NULL )     // user error
            {
                this->errno = EINVAL;
                return -1;
            }
            else                          // success
            {
                // get semaphore current value
		        remote_sem_get_value( sem_xp , &data );
 
                // return value to user
                hal_copy_to_uspace( value , &data , sizeof(uint32_t) );
            }
            break;
        }
        //////////////
	    case SEM_WAIT:
        {  
            // get extended pointer on remote semaphore
            xptr_t sem_xp = remote_sem_from_vaddr( (intptr_t)vaddr );

            if( sem_xp == XPTR_NULL )     // user error
            {
                this->errno = EINVAL;
                return -1;
            }
            else                          // success
            {
                // wait semaphore available
                remote_sem_wait( sem_xp );
            }
            break;
        }
        //////////////
	    case SEM_POST:
        {
            // get extended pointer on remote semaphore
            xptr_t sem_xp = remote_sem_from_vaddr( (intptr_t)vaddr );

            if( sem_xp == XPTR_NULL )     // user error
            {
                this->errno = EINVAL;
                return -1;
            }
            else                          // success
            {
                // release semaphore 
                remote_sem_post( sem_xp );
            }
			break;
        }
        /////////////////
	    case SEM_DESTROY:
        {
            // get extended pointer on remote semaphore
            xptr_t sem_xp = remote_sem_from_vaddr( (intptr_t)vaddr );

            if( sem_xp == XPTR_NULL )     // user error
            {
                this->errno = EINVAL;
                return -1;
            }
            else                          // success 
            {
                // destroy semaphore
                remote_sem_destroy( sem_xp );
            }
            break;
	    }	
        ///////
	    default:  // undefined operation                       
        {
		    this->errno = EINVAL;
            return -1;
        }
	}

    return 0;

}  // end sys_sem()
