/*
 * soclib_mmc.c - soclib simple block device driver implementation.
 *
 * Author     Alain Greiner (2016)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH..
 *
 * ALMOS-MKH. is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH. is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH.; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */


#include <hal_types.h>
#include <device.h>
#include <dev_mmc.h>
#include <soclib_mmc.h>
#include <spinlock.h>
#include <thread.h>
#include <printk.h>


/////////////////////////////////////
void soclib_mmc_init( xptr_t dev_xp )
{
    // get local pointer on MMC device descriptor
    device_t * dev_ptr = (device_t *)GET_PTR( dev_xp );

    // get extended pointer on MMC peripheral from MMC device
    xptr_t  mmc_xp = dev_ptr->base;

    // get local pointer on MMC segment base
    uint32_t * base = (uint32_t *)GET_PTR( mmc_xp );

    // enable MMC IRQ
    *(base + (SOCLIB_MMC_ERROR_FUNC << 7) + SOCLIB_MMC_ERROR_IRQ_ENABLE) = 1;
} 


//////////////////////////////////////////////////////////////////
void __attribute__ ((noinline)) soclib_mmc_command( xptr_t th_xp )
{
    xptr_t     dev_xp;       // extended pointer on MMC device
    uint32_t   type;         // MMC command : type
    uint64_t   buf_paddr;    // MMC command : buffer physical address
    uint32_t   buf_size;     // MMC command : buffer size
    uint32_t   reg_index;    // MMC command : register index in MMC peripheral
    uint32_t * reg_ptr;      // MMC command : pointer on dst/src buffer in client cluster 

    // get client thread cluster and local pointer
    cxy_t      th_cxy = GET_CXY( th_xp );
    thread_t * th_ptr = (thread_t *)GET_PTR( th_xp );

    // get command type and extended pointer on MMC device
    type   =         hal_remote_lw ( XPTR( th_cxy , &th_ptr->dev.mmc.type   ) );
    dev_xp = (xptr_t)hal_remote_lwd( XPTR( th_cxy , &th_ptr->dev.mmc.dev_xp ) );

    // get MMC device cluster and local pointer
    cxy_t      dev_cxy = GET_CXY( dev_xp );
    device_t * dev_ptr = (device_t *)GET_PTR( dev_xp );

    // get extended pointer on SOCLIB-MMC peripheral
    xptr_t     mmc_xp = hal_remote_lw( XPTR( dev_cxy , &dev_ptr->base ) );

    // get SOCLIB_MMC peripheral cluster and local pointer
    cxy_t      mmc_cxy = GET_CXY( mmc_xp );
    uint32_t * mmc_ptr = (uint32_t *)GET_PTR( mmc_xp );

    if( (type == MMC_CC_INVAL) || (type == MMC_CC_SYNC) )
    {
        // get buffer paddr
        buf_paddr = hal_remote_lwd( XPTR( th_cxy , &th_ptr->dev.mmc.buf_paddr ) );

        // split buffer paddr in two 32 bits words
        uint32_t   buf_lo = (uint32_t)( buf_paddr );
        uint32_t   buf_hi = (uint32_t)( buf_paddr>>32 );

        // get buffer size
        buf_size   = hal_remote_lw( XPTR( th_cxy , &th_ptr->dev.mmc.buf_size ) );

        // get command type
        uint32_t cc_cmd = MMC_CC_INVAL ? SOCLIB_MMC_CC_INVAL : SOCLIB_MMC_CC_SYNC; 

        // set SOCLIB_MMC registers to start INVAL/SYNC operation
        hal_remote_sw( XPTR( mmc_cxy , mmc_ptr + SOCLIB_MMC_ADDR_LO    ) , buf_lo );
        hal_remote_sw( XPTR( mmc_cxy , mmc_ptr + SOCLIB_MMC_ADDR_HI    ) , buf_hi );
        hal_remote_sw( XPTR( mmc_cxy , mmc_ptr + SOCLIB_MMC_BUF_LENGTH ) , buf_size ); 
        hal_remote_sw( XPTR( mmc_cxy , mmc_ptr + SOCLIB_MMC_CMD_TYPE   ) , cc_cmd ); 
    }
    else  // (type == MMC_GET_ERROR) or (type == MMC_GET_ERROR) pr (type == MMC_GET_INSTRU )
    {
        // get src/dst buffer local pointer and register index
        reg_ptr   = (uint32_t *)hal_remote_lpt( XPTR( th_cxy , &th_ptr->dev.mmc.reg_ptr ) );
        reg_index = hal_remote_lw( XPTR( th_cxy , &th_ptr->dev.mmc.reg_index ) );

        // move register to/from local buffer 
        if( (type == MMC_GET_ERROR) || (type == MMC_GET_INSTRU) )
        {
            *reg_ptr =  hal_remote_lw( XPTR( mmc_cxy , mmc_ptr + reg_index ) );
        }
        else  // type == MMC_SET_ERROR
        {
            hal_remote_sw( XPTR( mmc_cxy , mmc_ptr + reg_index ) , *reg_ptr );
        }
    }
} // end soclib_mmc_cmd()


////////////////////////////////////////////////////////////////
void __attribute__ ((noinline)) soclib_mmc_isr( device_t * dev )
{
    // get extended pointer on MMC peripheral from MMC device
    xptr_t  mmc_xp = dev->base;

    // get local pointer on MMC segment base
    uint32_t * base = (uint32_t *)GET_PTR( mmc_xp );

    // get faulty ADDRESS and SRCID from MMC registers
    uint32_t paddr_lo = *(base + (SOCLIB_MMC_ERROR_FUNC << 7) + SOCLIB_MMC_ERROR_ADDR_LO);
    uint32_t paddr_hi = *(base + (SOCLIB_MMC_ERROR_FUNC << 7) + SOCLIB_MMC_ERROR_ADDR_HI);
    uint32_t srcid    = *(base + (SOCLIB_MMC_ERROR_FUNC << 7) + SOCLIB_MMC_ERROR_SRCID);

    paddr_t paddr = (((paddr_t)paddr_hi)<<32) + ((paddr_t)paddr_lo);
    
    // reset MMC IRQ
    *(base + (SOCLIB_MMC_ERROR_FUNC << 7) + SOCLIB_MMC_ERROR_IRQ_RESET) = 0;
    
    // print an error message on kernel terminal TODO : should be improved
    printk("\n[ERROR] reported from MMC in cluster %x : faulty address = %l / srcid = %x\n",
           paddr , srcid );

} // end soclib_mmc_isr()



