/*
 * thread.h -  Thread and related operations definition.
 * 
 * Author  Ghassan Almaless (2008,2009,2010,2011,2012)
 *         Mohamed Lamine Karaoui (2015)
 *         Alain Greiner (2016)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef _THREAD_H_
#define _THREAD_H_

#include <hal_types.h>
#include <hal_special.h>
#include <xlist.h>
#include <list.h>
#include <hal_context.h>
#include <spinlock.h>
#include <core.h>
#include <cluster.h>
#include <process.h>
#include <dev_ioc.h>
#include <dev_nic.h>
#include <dev_txt.h>
#include <dev_mmc.h>

/***************************************************************************************
 * This defines the various pthread_attr_t flags.
 **************************************************************************************/

#define PT_FLAG_DETACH                0x001    // user defined not joinable
#define PT_FLAG_CLUSTER_DEFINED       0x002    // user defined target cluster
#define PT_FLAG_CORE_DEFINED          0x004    // user defined core cluster

/***************************************************************************************
 * This structure defines the input argument of the pthread_create() system call.
 * It contains all informations required to initialize an user thread, and is passed
 * as input argument to the thread_user_create() function.
 * It is partly set by the kernel, partly set by the user application itself,
 * using the pthread_attr_***() functions.
 **************************************************************************************/
 
typedef struct pthread_attr_s
{
    pid_t       pid;             /*! owner process identifier                         */
	void      * entry_func;      /*! pointer on entry function                        */
	void      * entry_args;      /*! pointer on entry function arguments              */
	uint32_t    flags;           /*! pthread flags        (entirely user specified)   */
	cxy_t       cxy;             /*! target cluster       (can be user specified)     */
	lid_t       lid;             /*! target core          (can be user specified)     */
} 
pthread_attr_t;


/***************************************************************************************
 * This enum defines the thread types.
 **************************************************************************************/

typedef enum
{   
	THREAD_USER    = 0,          /*! user thread (pthread)                            */ 
	THREAD_RPC     = 1,          /*! kernel thread executing pending RPCs             */
	THREAD_DEV     = 2,          /*! kernel thread executing I/O device commands      */
	THREAD_KERNEL  = 3,          /*! other kernel thread                              */
	THREAD_IDLE    = 4,          /*! kernel idle thread                               */
	THREAD_TYPES_NR
} 
thread_type_t;

/***************************************************************************************
 * This defines the masks associated to the thread flags.
 **************************************************************************************/

#define THREAD_FLAG_LOADABLE     0x0001  /*! This thread has not been executed yet    */
#define THREAD_FLAG_DETACHED     0x0002  /*! This user thread is detached from parent */

/***************************************************************************************
 * This defines the masks associated to the thread signals.
 **************************************************************************************/

#define THREAD_SIG_KILL          0x0001  /*! This thread must be destroyed ASAP       */

/***************************************************************************************
 * This defines the masks associated to the blocking causes.
 **************************************************************************************/

#define THREAD_BLOCKED_GLOBAL    0x0001  /*! thread global blocking                   */ 
#define THREAD_BLOCKED_IO        0x0002  /*! thread wait IO operation completion      */
#define THREAD_BLOCKED_MAPPER    0x0004  /*! thread wait mapper                       */
#define THREAD_BLOCKED_EXIT      0x0010  /*! thread requested exit                    */
#define THREAD_BLOCKED_KILL      0x0020  /*! thread received kill signal              */
#define THREAD_BLOCKED_SEM       0x0040  /*! thread wait semaphore                    */
#define THREAD_BLOCKED_PAGE      0x0080  /*! thread wait page access                  */
#define THREAD_BLOCKED_IDLE      0x1000  /*! thread RPC wait activation               */
#define THREAD_BLOCKED_DEV_QUEUE 0x2000  /*! thread DEV wait queue                    */
#define THREAD_BLOCKED_DEV_ISR   0x4000  /*! thread DEV wait ISR                      */

/***************************************************************************************
 * This structure defines thread instrumentation informations.
 **************************************************************************************/

typedef struct thread_info_s
{
	uint32_t              pgfault_nr;    /*! cumulated number of page fault           */
	uint32_t              sched_nr;      /*! TODO ???  [AG]                           */
	uint32_t              u_err_nr;      /*! TODO ???  [AG]                           */
	uint32_t              m_err_nr;      /*! TODO ???  [AG]                           */
	uint32_t              tm_tmp;        /*! temp date to compute execution duration  */
	uint32_t              tm_exec;       /*! TODO ???  [AG]                           */
	uint32_t              tm_create;     /*! date of the creation                     */
	uint32_t              tm_born;       /*! date of the thread loading               */
	uint32_t              tm_dead;       /*! date of the death                        */
	clock_t               tm_sleep;      /*! TODO ???  [AG]                           */
	clock_t               tm_wait;       /*! TODO ???  [AG]                           */
	clock_t               tm_usr;        /*! user execution duration                  */
	clock_t               tm_sys;        /*! system execution duration                */
}
thread_info_t;

/***************************************************************************************
 * This structure defines a thread descriptor.
 * It is used for both the user threads and the kernel threads.
 * In a process, an user thread is identified by a unique TRDID (thread identifier),
 * that is returned by the kernel to the user:
 * - The TRDID 16 LSB bits contain the LTID (Local Thread Index).
 * - The TRDID 16 MSB bits contain the owner cluster CXY.
 * - The LTID is used to index the th_tbl[] array in the local process descriptor.
 * This TRDID is computed by the process_register_thread() function, when the user
 * thread is registerd in the local copy of the process descriptor.
 **************************************************************************************/

#define THREAD_SIGNATURE    0xDEADBEEF

typedef struct thread_s
{
	uint32_t            trdid;           /*! thread index (in THTBL)                  */
	thread_type_t       type;            /*! thread type                              */
	uint32_t            quantum;         /*! number of clock ticks given to thread    */
	uint32_t            ticks_nr;        /*! number of ticks used                     */
	uint32_t            time_last_check; /*! last cpu_time_stamp                      */
	core_t            * core;            /*! pointer to the owner core                */
	process_t         * process;         /*! pointer on local process descriptor      */
	page_t            * page;            /*! pointer on page desc. containing thread  */ 

	uint32_t            local_locks;	 /*! number of local locks owned by thread    */
    list_entry_t        locks_root;      /*! root of local locks list                 */

	uint32_t            remote_locks;	 /*! number of local locks owned by thread    */
    xlist_entry_t       xlocks_root;     /*! root of remote locks list                */

	intptr_t            u_stack_base;    /*! user stack base address                  */
	uint32_t            u_stack_size;    /*! user stack size (bytes)                  */
	intptr_t            k_stack_base;    /*! kernel stack base address                */
	uint32_t            k_stack_size;    /*! kernel stack size (bytes)                */

    void              * entry_func;      /*! pointer on entry function                */
    void              * entry_args;      /*! pointer on entry function arguments      */

    uint32_t            flags;           /*! bit vector of flags                      */
    uint32_t            blocked;         /*! bit vector of blocking causes            */
    uint32_t            signals;         /*! bit vector of signals                    */

	error_t             errno;           /*! errno value of last system call          */

    bool_t              fork_user;       /*! user defined placement for next fork()   */
    cxy_t               fork_cxy;        /*! target cluster  for next fork()          */

	xlist_entry_t       children_root;   /*! root of list of attached children        */
    uint32_t            children_nr;     /*! number of attached children threads      */
    remote_spinlock_t * children_lock;   /*! lock protecting the children list        */

    xlist_entry_t       brothers_list;   /*! member of threads with same parent       */

	list_entry_t        sched_list;      /*! member of threads attached to same core  */

    uint32_t            dev_channel;     /*! device channel for a DEV thread          */
    union                                /*! embedded command for a DEV thread        */
    {
        ioc_command_t   ioc;             /*! IOC device generic command               */
        txt_command_t   txt;             /*! TXT device generic command               */
        nic_command_t   nic;             /*! NIC device generic command               */
        mmc_command_t   mmc;             /*! NIC device generic command               */
    }  
    dev;
    
	cxy_t               rpc_client_cxy;  /*! client cluster index (for a RPC thread)  */

    xlist_entry_t       wait_list;       /*! member of threads blocked on same cond   */

	void              * fpu_context;     /*! pointer on FPU context                   */
	void              * cpu_context;     /*! pointer on CPU context                   */

	thread_info_t       info;            /*! embedded thread_info_t                   */

	uint32_t            signature;       /*! for kernel stack overflow detection      */
} 
thread_t;

/***************************************************************************************
 * This macro returns a pointer on the calling thread from the core hardware register.
 **************************************************************************************/

#define CURRENT_THREAD  (hal_get_current_thread())

/***************************************************************************************
 * This function allocates memory for an user thread descriptor in the local cluster,
 * and initializes it from information contained in the "attr" argument.
 * It is used by the pthread_create system call, the CPU context is initialised from
 * scratch, and the "loadable" field is set.
 * The new thread is attached to the core specified in the "attr" argument.
 * It is registered in the local process descriptor for the process specified in "attr".
 * The thread descriptor pointer is returned to allow the parent thread to register it
 * in its children list.
 * The THREAD_BLOCKED_GLOBAL bit is set, and the thread must be activated to start.   
 ***************************************************************************************
 * @ new_thread   : address of buffer for new thread descriptor pointer.  
 * @ attr         : pointer on pthread attributes descriptor.
 * @ u_stack_base : actual user stack size.
 * @ u_stack_size : actual user stack base.
 * @ returns 0 if success / returns ENOMEM if error.
 **************************************************************************************/
error_t thread_user_create( thread_t       ** new_thread,
                            pthread_attr_t  * attr,
                            intptr_t          u_stack_base,
                            uint32_t          u_stack_size );

/***************************************************************************************
 * This function allocates memory for an user thread descriptor in the local cluster,
 * and initialises it from informations contained in the calling thread descriptor.
 * It is used by the fork() system call to create the child process main thread.
 * The new thread is attached to the core with the lowest load. 
 * It is registered in the process descriptor defined by the "process" argument.
 * This new thread inherits its execution context from the calling thread,
 * and the "loadable" field is NOT set.
 * The THREAD_BLOCKED_GLOBAL bit is set, and the thread must be activated to start.   
 ***************************************************************************************
 * @ new_thread   : address of buffer for new thread descriptor pointer.  
 * @ process      : local pointer on owner process descriptor.
 * @ u_stack_base : actual user stack size.
 * @ u_stack_size : actual user stack base.
 * @ returns 0 if success / returns ENOMEM if error.
 **************************************************************************************/
error_t thread_user_fork( thread_t ** new_thread,
                          process_t * process,
                          intptr_t    u_stack_base,
                          uint32_t    u_stack_size );

/***************************************************************************************
 * This function allocates memory for a kernel thread descriptor in the local cluster,
 * and initialise it from arguments values.
 * The THREAD_BLOCKED_GLOBAL bit is set, and the thread must be activated to start.   
 ***************************************************************************************
 * @ new_thread   : address of buffer for new thread pointer.
 * @ type         : kernel thread type.
 * @ func         : pointer on function.
 * @ args         : function arguments.
 * @ core_lid     : local core index.
 * @ returns 0 if success / returns ENOMEM if error
 **************************************************************************************/
error_t thread_kernel_create( thread_t     ** new_thread,
                              thread_type_t   type,
                              void          * func, 
                              void          * args,
                              lid_t           core_lid );

/***************************************************************************************
 * This function releases the physical memory allocated for a thread descriptor 
 * in the local cluster. It can be used for both an user and a kernel thread.
 * The physical memory dynamically allocated in the HEAP or MMAP zones by an user 
 * thread will be released when the process is killed, and the page table flushed.
 ***************************************************************************************
 * @ thread  : pointer on the thread descriptor to release.  
 **************************************************************************************/
void thread_destroy( thread_t * thread );

/***************************************************************************************
 * This function defines the code of the thread executed when no other thread is 
 * runnable for a given core. It try to force the core to the low-power state.
 **************************************************************************************/
void * thread_idle_func();

/***************************************************************************************
 * This function registers a child thread in the global list of attached 
 * children threads of a parent thread.
 * It does NOT take a lock, as this function is allways called by the parent thread.
 ***************************************************************************************
 * @ xp_parent : extended pointer on the parent thread descriptor.
 * @ xp_child  : extended pointer on the child thread descriptor.
 **************************************************************************************/
void thread_child_parent_link( xptr_t  xp_parent,
                               xptr_t  xp_child );

/***************************************************************************************
 * This function removes an user thread from the parent thread global list 
 * of attached children threads. 
 * It does take the lock, as this function can be called by the child thread.
 * In this case, it uses the extended pointer on the parent thread contained 
 * in the pthread_attr_t embedded in the child thread descriptor.
 ***************************************************************************************
 * @ xp_parent : extended pointer on the parent thread descriptor.
 * @ xp_child  : extended pointer on the child thread descriptor.
 **************************************************************************************/
void thread_child_parent_unlink( xptr_t xp_parent,
                                 xptr_t xp_child );


/***************************************************************************************
 * This function atomically set a signal in a thread descriptor.
 ***************************************************************************************
 * @ thread    : local pointer on target thread.
 * @ mask      : mask on selected signal.
 **************************************************************************************/
inline void thread_set_signal( thread_t * thread,
                               uint32_t   mask );

/***************************************************************************************
 * This function reset a signal in a thread descriptor.
 ***************************************************************************************
 * @ thread    : local pointer on target thread.
 * @ mask      : mask on selected signal.
 **************************************************************************************/
inline void thread_reset_signal( thread_t * thread,
                                 uint32_t   mask );

/***************************************************************************************
 * This function returns true if the calling thread is attached to its parent thread.
 **************************************************************************************/
inline bool_t thread_is_joinable();

/***************************************************************************************
 * This function returns true if the calling thread is not blocked.
 **************************************************************************************/
inline bool_t thread_is_runnable();

/***************************************************************************************
 * This function cheks if the calling thread can deschedule.
 ***************************************************************************************
 * @ returns true if no locks taken.
 **************************************************************************************/
inline bool_t thread_can_yield();

/***************************************************************************************
 * This function checks if the calling thread must be descheduled. 
 ***************************************************************************************
 * @ returns true if no locks taken, and elapsed time.
 **************************************************************************************/
bool_t thread_check_sched();

/***************************************************************************************
 * This function can be used by the calling thread to suicide.
 * All locks must be previouly released.
 * The scenario depends on the attached/detached flag :
 * - if detached, it set the SIG_KILL signal in the "signals" bit_vector, registers
 *   the BLOCKED_EXIT bit in the "blocked" bit_vector, and deschedule.
 * - if attached, it simply set the BLOCKED_EXIT bit in the "blocked" bit vector
 *   and deschedule. The SIG_KILL signal will be set by the parent thread when
 *   executing the pthread_join().
 ***************************************************************************************
 * @ returns 0 if success / returns EINVAL if locks_count is not zero.
 **************************************************************************************/
error_t thread_exit();

/***************************************************************************************  
 * This function registers a blocking cause in the target thread "blocked" bit vector.
 * Warning : this function does not deschedule the calling thread.
 * The descheduling can be forced by a sched_yield(). 
 ***************************************************************************************
 * @ thread   : local pointer on target thread.
 * @ cause    : mask defining the cause (one hot).
 **************************************************************************************/
void thread_block( thread_t * thread,
                   uint32_t   cause );

/***************************************************************************************  
 * This function reset the bit identified by the cause argument in the "blocked"
 * bit vector of a remote thread descriptor.
 * We need an extended pointer, because the client thread of an I/O operation on a
 * given device is not in the same cluster as the associated device descriptor.
 * Warning : this function does not reschedule the remote thread. 
 * The scheduling can be forced by sending an IPI to the core running the remote thread.
 ***************************************************************************************
 * @ thread   : extended pointer on the remote thread.
 * @ cause    : mask defining the cause (one hot).
 **************************************************************************************/
void thread_unblock( xptr_t   thread,
                     uint32_t cause );

/***************************************************************************************  
 * This function kills a target thread, identified by its local pointer.
 * It is generally called by the local process_destroy() function.
 * - it forces the global blocked bit in target thread descriptor.
 * - it set the SIG_KILL signal in target thread.
 * - it send an IPI_SCHED_REQUEST to the target thread core.
 ***************************************************************************************
 * @ thread   : local pointer on the target thread.
 * @ returns 0 if success / returns EINVAL if locks_count is not zero.
 **************************************************************************************/
void thread_kill( thread_t * thread );

#endif	/* _THREAD_H_ */
