/*
 * dev_mwr.h - MWR (Multi Fifos DMA Controller) generic device API.
 *
 * Authors   Alain Greiner  (2016)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTMWRLAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef _DEV_MWR_H_
#define _DEV_MWR_H_

#include <almos_config.h>
#include <hal_types.h>
#include <spinlock.h>

/*****************************************************************************************
 *     Generic multi-fifos DMA controller for harware coprocessors.
 *
 ****************************************************************************************/
 
struct device_s;

/*****************************************************************************************
 * This defines the (implementation independant) extension for the generic MWR device.
 ****************************************************************************************/

typedef struct mwr_extend_s
{
    uint32_t    to_coproc_nr;    /*! actual number of to_coproc FIFOs                   */
    uint32_t    from_coproc_nr;  /*! actual number of from coproc FIFOs                 */
    uint32_t    config_nr;       /*! actual number of configuration registers           */
    uint32_t    status_nr;       /*! actual number of status registers                  */
}
mwr_extend_t;


/*****************************************************************************************
 * This function initializes the MWR specific fields of the device descriptor.
 * It must be executed once in the cluster containing a MWR device descriptor.
 *****************************************************************************************
 * @ mwr_xp         : extended pointer on MWR device descriptor.
 * @ to_coproc_nr   : number of to_coproc fifos.
 * @ from_coproc_nr : number of from_coproc fifos.
 * @ config_nr      : number of coprocessor configuration registers.
 * # status_nr      : number of coprocessor status registers.
 ****************************************************************************************/
void dev_mwr_init( xptr_t    mwr_xp,
                   uint32_t  to_coproc_nr,
                   uint32_t  from_coproc_nr,
                   uint32_t  config_nr,
                   uint32_t  status_nr );

/*****************************************************************************************
 * This function set a value in one coprocessor register.
 * It can only be executed by a local thread.
 *****************************************************************************************
 * @ cxy     : MMC cluster identifier.
 * @ index   : register index in MMC peripheral.
 * @ wdata   : value to be written.
 ****************************************************************************************/
void dev_mwr_set_coproc_register( cxy_t    cxy,
                                  uint32_t index,
                                  uint32_t wdata );

/*****************************************************************************************
 * This function returns the value contained in one coprocessor register.
 * It can only be executed by a local thread.
 *****************************************************************************************
 * @ cxy     : MMC cluster identifier.
 * @ index   : error register index in MMC peripheral.
 * @ rdata   : local pointer on buffer for returned value.
 ****************************************************************************************/
void dev_mmc_get_coproc_register( cxy_t      cxy,
                                  uint32_t   index, 
                                  uint32_t * rdata );

/*****************************************************************************************
 * This function set a value in one channel register.
 * It can only be executed by a local thread.
 *****************************************************************************************
 * @ cxy     : MMC cluster identifier.
 * @ index   : register index in MMC peripheral.
 * @ wdata   : value to be written.
 ****************************************************************************************/
void dev_mwr_set_channel_register( cxy_t    cxy,
                                   uint32_t channel,
                                   uint32_t index,
                                   uint32_t wdata );

/*****************************************************************************************
 * This function returns the value contained in one channel register.
 * It can only be executed by a local thread.
 *****************************************************************************************
 * @ cxy     : MMC cluster identifier.
 * @ index   : error register index in MMC peripheral.
 * @ rdata   : local pointer on buffer for returned value.
 ****************************************************************************************/
void dev_mmc_get_channel_register( cxy_t      cxy,
                                   uint32_t channel,
                                   uint32_t   index, 
                                   uint32_t * rdata );



#endif  /* _DEV_MWR_H_ */
