/*
 * dev_pic.c - PIC (External Interrupt Controler) generic device API implementation.
 *
 * Authors   Alain Greiner  (2016)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH.is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH.is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTPICLAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH.; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_types.h>
#include <hal_special.h>
#include <dev_icu.h>
#include <device.h>
#include <memcpy.h>
#include <printk.h>
#include <soclib_pic.h>
#include <dev_pic.h>

/////////////////////////////////////////////////////////////////////////////////////////
// Extern global variables
/////////////////////////////////////////////////////////////////////////////////////////

extern devices_directory_t  devices_dir;         // allocated in kernel_init.c

extern devices_input_irq_t  devices_input_irq;   // allocated in kernel_init.c



///////////////////////////////////
void dev_pic_init( xptr_t   dev_xp,
                   uint32_t irq_nr )
{
    // get PIC device cluster and local pointer
    cxy_t      dev_cxy = GET_CXY( dev_xp );
    device_t * dev_ptr = (device_t *)GET_PTR( dev_xp );

    // set PIC device extension field
    dev_ptr->ext.pic.irq_nr = irq_nr;

    // get implementation index from PIC device descriptor
    uint32_t impl = hal_remote_lw( XPTR( dev_cxy , &dev_ptr->impl ) );

    // set the "name" field in PIC device descriptor
    // and call the relevant driver init function
    if( impl == IMPL_PIC_SOC )
    {
        memcpy( dev_ptr->name , "PIC_TSR" , 16 );
        soclib_pic_init( dev_xp );
    }
    else
    {
        printk("PANIC in %s: illegal PIC device implementation\n", __FUNCTION__ );
        hal_core_sleep();
    }
} // end dev_pic_init()

/////////////////////////////////////////
void dev_pic_bind_irq( uint32_t   irq_id,
                       cxy_t      cxy,
                       uint32_t   wti_id )
{
    // get extended pointer on WTI mailbox
    xptr_t wti_xp = dev_icu_wti_xptr( cxy , wti_id );

    // get extended pointer on PIC device from directory
    xptr_t dev_xp = devices_dir.pic;

    // get PIC device cluster and local pointer
    cxy_t      dev_cxy = GET_CXY( dev_xp );
    device_t * dev_ptr = (device_t *)GET_PTR( dev_xp );

    // get implementation index and segment base 
    uint32_t impl   = hal_remote_lw( XPTR( dev_cxy , &dev_ptr->impl ) );

    // call the implementation specific driver function
    if( impl == IMPL_PIC_SOC )
    {
        soclib_pic_bind_irq( dev_xp , irq_id , wti_xp );
    }
    else
    {
        printk("PANIC in %s: illegal PIC device implementation\n", __FUNCTION__ );
        hal_core_sleep();
    }
}  // end dev_pic_link_wti()

//////////////////////////////////////////
void dev_pic_unbind_irq( uint32_t irq_id )
{
    // get extended pointer on PIC device from directory
    xptr_t dev_xp = devices_dir.pic;

    // get PIC device cluster and local pointer
    cxy_t      dev_cxy = GET_CXY( dev_xp );
    device_t * dev_ptr = (device_t *)GET_PTR( dev_xp );

    // get implementation index 
    uint32_t impl = hal_remote_lw( XPTR( dev_cxy , &dev_ptr->impl ) );

    // call the implementation specific driver function
    if( impl == IMPL_PIC_SOC )
    {
        soclib_pic_unbind_irq( dev_xp , irq_id );
    }
    else
    {
        printk("PANIC in %s: illegal PIC device implementation\n", __FUNCTION__ );
        hal_core_sleep();
    }
   
} // end dev_pic_disable_irq()


