/*
 * soclib_xcu.c - x86 XCU driver API implementation.
 *
 * Copyright (c) 2017 Maxime Villard
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH.; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <soclib_xcu.h>
#include <hal_types.h>
#include <core.h>
#include <chdev.h>

#include <hal_apic.h>
#include <hal_segmentation.h>
#include <hal_internal.h>

extern size_t ioapic_pins;

/*
 * These indexes are used to translate a type::idx to a pin on the IOAPIC.
 */
uint32_t hwi_baseidx __in_kdata = 0;
uint32_t wti_baseidx __in_kdata = 0;
uint32_t pti_baseidx __in_kdata = 0;

static uint32_t get_pin(uint32_t idx, uint32_t type)
{
	switch (type) {
		case HWI_TYPE:
			return hwi_baseidx + idx;
		case WTI_TYPE:
			return wti_baseidx + idx;
		case PTI_TYPE:
			return pti_baseidx + idx;
		default:
			x86_panic("get_pin: wrong type");
			return 0;
	}
}

void soclib_xcu_init(chdev_t *icu, lid_t lid)
{
	size_t i;

	/* disable all IRQs */
	for (i = 0; i < ioapic_pins; i++) {
		hal_ioapic_disable_entry(i);
	}

	x86_panic((char *)__func__);
}

void soclib_xcu_enable_irq(chdev_t *icu, uint32_t idx, uint32_t type,
    lid_t lid)
{
	uint8_t dest = (uint8_t)lid; /* XXX */
	uint32_t pin;
	uint8_t vec;

	pin = get_pin(idx, type);

	switch (type) {
		case HWI_TYPE:
			vec = VECTOR_APIC_XCU_HWI;
			break;
		case WTI_TYPE:
			vec = VECTOR_APIC_XCU_WTI;
			break;
		case PTI_TYPE:
			vec = VECTOR_APIC_XCU_PTI;
			break;
		default:
			x86_panic("enabling wrong irq");
	}

	hal_ioapic_set_entry(pin, vec, dest);

	x86_panic((char *)__func__);
}

void soclib_xcu_disable_irq(chdev_t *icu, uint32_t idx, uint32_t type,
    lid_t lid)
{
	uint32_t pin = get_pin(idx, type);

	hal_ioapic_disable_entry(pin);

	x86_panic((char *)__func__);
}

void soclib_xcu_get_masks(chdev_t *icu, lid_t lid, uint32_t *hwi_mask,
    uint32_t *wti_mask, uint32_t *pti_mask)
{
	x86_panic((char *)__func__);
}

void soclib_xcu_set_period( chdev_t * icu,
                            uint32_t  index,
                            uint32_t  period )
{
	x86_panic((char *)__func__);
}

uint32_t soclib_xcu_ack_timer( chdev_t * icu,
                               uint32_t  index )
{
	x86_panic((char *)__func__);
	return 0;
}

void soclib_xcu_get_status(chdev_t *icu, lid_t lid, uint32_t *hwi_status,
    uint32_t *wti_status, uint32_t *pti_status)
{
	if (lid != 0) {
		x86_panic("xcu_get_status should have lid==0");
	}

	x86_panic((char *)__func__);
}

void soclib_xcu_send_ipi( xptr_t  icu_xp,
                          lid_t   lid )
{
	x86_panic((char *)__func__);
}

uint32_t * soclib_xcu_wti_ptr( chdev_t  * icu,
                               uint32_t   index )
{
	x86_panic((char *)__func__);
	return NULL;
}
