/*
 * hal_do_exception.c - Architecture specific exception handler API definition
 * 
 * Authors    Alain Greiner (2016,2017)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef _HAL_EXCEPTION_H_
#define _HAL_EXCEPTION_H_

#include <hal_types.h>

//////////////////////////////////////////////////////////////////////////////////////////
//     Architecture specific exception handler API
//
// ALMOS-MKH defines three classes of exceptions:
//
// - NON_FATAL : exceptions such as "page fault" or FPU unusable" are non fatal.
//   and the calling thread resume execution when the exception has been handled.
//
// - USER_ERROR : these exceptions such a "illegal vaddr" or "illegal write access"
//   are fatal. The calling thread process is killed, after an "error" message on the
//   kernel terminal.
//
// - KERNEL_PANIC : events such as "no memory" or "kernel mistakes" are considered
//   abnormal events. The calling core goes to sleep, after after a "panic" message
//   on the kernel terminal.
//
// For all exceptions, the faulty core context has been saved in a registers
// array stored in the user thread descriptor (core in user mode), and in the
// kernel stack (core in kernel mode). 
//
// Any architecture specific implementation must implement this API.
//////////////////////////////////////////////////////////////////////////////////////////

/**** forward declaration  ****/

struct thread_s;

/*****************************************************************************************
 * This function implements the ALMOS-MKH exception handler.
 * It is called by the hal_kentry function when an exception is detected
 * by the hardware for an user thread running on a given core.
 *****************************************************************************************
 * @ this      : pointer on the faulty thread descriptor.
 * @ regs_tbl  : array containing the core registers values saved by hal_kentry.
 ****************************************************************************************/
void hal_do_exception( struct thread_s * this, 
                       reg_t           * regs_tbl );

/*****************************************************************************************
 * This function print on the kernel terminal the saved context (core registers)
 * and the thread state of a faulty thread.
 *****************************************************************************************
 * @ this      : pointer on the faulty thread descriptor.
 * @ regs_tbl  : pointer on the array containing the core registers values.
 ****************************************************************************************/
void hal_exception_dump( struct thread_s * this,
                         reg_t           * regs_tbl );

#endif  // _HAL_EXCEPTION_H_
