/*
 * hal_uspace.h - Generic User Space Access API definition
 *
 * Authors   Mohamed Karaoui  (2015)
 *           Alain Greiner    (2016)
 *
 * Copyright (c)  UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef  _HAL_USPACE_H_
#define  _HAL_USPACE_H_

#include <hal_types.h>

//////////////////////////////////////////////////////////////////////////////////////////
//           User space access API (implementation in hal_uspace.c)
//
// When moving data between user space and kernel space, the user address is always
// a virtual address, but the kernel address can be a physical address, on some
// architectures. Therefore, data transfers must use the following API.
//////////////////////////////////////////////////////////////////////////////////////////


/*****************************************************************************************
 * This function tranfers a data buffer from the user space to the kernel space.
 * As the kernel is using physical addresses on the TSAR architecture, it activates
 * the MMU to access the user buffer.
 *****************************************************************************************
 * @ k_dst     : destination address in kernel space.
 * @ u_src     : source buffer address in user space.
 * @ size      : size (number of bytes).
 ****************************************************************************************/
extern void hal_copy_from_uspace( void     * k_dst,
                                  void     * u_src,
                                  uint32_t   size );

/*****************************************************************************************
 * This function tranfers a data buffer from the kernel space to the user space.
 * As the kernel is using physical addresses on the TSAR architecture, it activates
 * the MMU to access the user buffer.
 *****************************************************************************************
 * @ u_dst     : destination buffer address and size in user space.
 * @ k_src     : source address in kernel space.
 * @ size      : size (number of bytes).
 ****************************************************************************************/
extern void hal_copy_to_uspace( void     * udst,
                                void     * ksrc,
                                uint32_t   size );

/*****************************************************************************************
 * This function activates the MMU to compute the length of a string in user space,
 * and returns it to a kernel buffer.
 *****************************************************************************************
 * @ u_str     : string address in user space.
 * @ return length of the string.
 ****************************************************************************************/
uint32_t hal_strlen_from_uspace( char     * string );


#endif	/* _HAL_USPACE_H_ */
