/*
 * dev_pic.c - PIC (External Interrupt Controler) generic device API implementation.
 *
 * Authors   Alain Greiner  (2016)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH.is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH.is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTPICLAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH.; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_types.h>
#include <hal_special.h>
#include <chdev.h>
#include <printk.h>
#include <soclib_pic.h>
#include <dev_pic.h>
#include <cluster.h>

/////////////////////////////////////////////////////////////////////////////////////////
// Extern global variables
/////////////////////////////////////////////////////////////////////////////////////////

extern chdev_directory_t  chdev_dir;         // allocated in kernel_init.c

///////////////////////////////////
void dev_pic_init( chdev_t  * pic )
{
    // get implementation 
    uint32_t impl = pic->impl;

    // set chdev name
    strcpy( pic->name , "pic" );

    // call the relevant driver init function,
    // and register commands in PIC device extension
    if( impl == IMPL_PIC_SCL )
    {
        // call the PIC SOCLIB driver
        soclib_pic_init( pic );

        // update the PIC chdev extension
        pic->ext.pic.enable_timer = &soclib_pic_enable_timer;
        pic->ext.pic.enable_irq   = &soclib_pic_enable_irq;
        pic->ext.pic.disable_irq  = &soclib_pic_disable_irq;
        pic->ext.pic.bind_irq     = &soclib_pic_bind_irq;
        pic->ext.pic.send_ipi     = &soclib_pic_send_ipi;
        pic->ext.pic.extend_init  = &soclib_pic_extend_init;
    }
    else if( impl == IMPL_PIC_I86 )
    {
        assert( false , __FUNCTION__ , "missing implementation for X86\n" );
    }
    else
    {
        assert( false , __FUNCTION__ , "undefined PIC device implementation" );
    }
} // end dev_pic_init()

/////////////////////////////////////////////////
void dev_pic_extend_init( uint32_t * lapic_base )
{
    // get pointer on PIC chdev
    chdev_t * pic = (chdev_t *)GET_PTR( chdev_dir.pic );

    // call relevant driver function
    pic->ext.pic.extend_init( lapic_base );
}

/////////////////////////////////////
void dev_pic_bind_irq( lid_t     lid,
                       chdev_t * src_chdev )
{
    // get pointer on PIC chdev
    chdev_t * pic = (chdev_t *)GET_PTR( chdev_dir.pic );

    // call relevant driver function
    pic->ext.pic.bind_irq( lid , src_chdev );
}  

///////////////////////////////////////
void dev_pic_enable_irq( lid_t     lid,
                         chdev_t * src_chdev )
{
    // get pointer on PIC chdev
    chdev_t * pic = (chdev_t *)GET_PTR( chdev_dir.pic );

    // call relevant driver function
    pic->ext.pic.enable_irq( src_chdev );
}

////////////////////////////////////////
void dev_pic_disable_irq( lid_t     lid,
                          chdev_t * src_chdev )
{
    // get pointer on PIC chdev
    chdev_t * pic = (chdev_t *)GET_PTR( chdev_dir.pic );

    // call relevant driver function
    pic->ext.pic.disable_irq( src_chdev );
}

////////////////////////////////////////////
void dev_pic_enable_timer( uint32_t period )
{
    // get pointer on PIC chdev
    chdev_t * pic = (chdev_t *)GET_PTR( chdev_dir.pic );

    // call relevant driver function
    pic->ext.pic.enable_timer( period );
}

//////////////////////////////////
void dev_pic_send_ipi( cxy_t  cxy,
                       lid_t  lid )
{
    // get pointer on PIC chdev
    chdev_t * pic = (chdev_t *)GET_PTR( chdev_dir.pic );

    // call relevant driver function
    pic->ext.pic.send_ipi( cxy , lid );
}


