/*
 * remote_spinlock.h - kernel remote spinlock definition.
 *
 * Authors  Mohamed Karaoui (2016)
 *          Alain Greiner   (2016)
 * 
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef _REMOTE_SPINLOCK_H_
#define _REMOTE_SPINLOCK_H_

#include <almos_config.h>
#include <hal_types.h>
#include <xlist.h>

/***************************************************************************************
 * This structure defines a remote spinlock, that can be used to protect
 * exclusive access to a trans-cluster shared resource. It can be taken by any 
 * thread running in any cluster. All access functions use remote pointers,
 * and the owner thread is registrated as a remote pointer.
 **************************************************************************************/

typedef struct remote_spinlock_s
{
    volatile uint32_t     taken;       /*! free if 0 / taken if non zero             */
    xptr_t                owner;       /*! extended pointer on the owner thread      */
    xlist_entry_t         list;        /*! list of all remote_lock taken by owner    */
} 
remote_spinlock_t;

/***************************************************************************************
 * This function initializes a remote spinlock.
 ***************************************************************************************
 * @ lock    : extended pointer on the remote spinlock
 **************************************************************************************/
void remote_spinlock_init( xptr_t   lock );

/***************************************************************************************
 * This blocking function locks a remote spinlock.
 * If the lock is already taken, the calling thread deschedule, and retry
 * when it is rescheduled.
 * It increments the calling thread locks count when the lock has been taken.
 ***************************************************************************************
 * @ lock    : extended pointer on the remote spinlock
 **************************************************************************************/
void remote_spinlock_lock( xptr_t lock );

/***************************************************************************************
 * This non blocking function try once to lock a remote spinlock.
 * It increments the calling thread locks count in case of success.
 ***************************************************************************************
 * @ lock    : extended pointer on the remote spinlock
 * @ returns O if success / returns non zero if lock already taken.
 **************************************************************************************/
uint32_t remote_spinlock_trylock( xptr_t lock );

/***************************************************************************************
 * This function releases a remote spinlock.
 ***************************************************************************************
 * @ lock    : extended pointer on the remote spinlock
 **************************************************************************************/
void remote_spinlock_unlock( xptr_t lock );

#endif
