/*
 * process.h - process related management functions
 * 
 * Authors  Ghassan Almaless (2008,2009,2010,2011,2012)
 *          Mohamed Lamine Karaoui (2015)
 *          Alain Greiner (2016)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef _PROCESS_H_
#define _PROCESS_H_

#include <almos_config.h>
#include <errno.h>
#include <hal_types.h>
#include <list.h>
#include <xlist.h>
#include <bits.h>
#include <spinlock.h>
#include <hal_atomic.h>
#include <vmm.h>
#include <signal.h>
#include <cluster.h>
#include <vfs.h>

/****  Forward declarations  ****/

struct thread_s;

/*********************************************************************************************
 * These macros are used to compose or decompose global process identifier (PID)
 * to or from cluster identifier / local process index (CXY , LPID)
 ********************************************************************************************/

#define LPID_FROM_PID( pid )       (lpid_t)(pid & 0x0000FFFF)
#define CXY_FROM_PID( pid )        (cxy_t)(pid >> 16)
#define PID( cxy , lpid )          (pid_t)((cxy << 16) | lpid )

/*********************************************************************************************
 * These macros are used to compose or decompose global thread identifier (TRDID)
 * to or from cluster identifier / local thread index (CXY , LTID)
 ********************************************************************************************/

#define LTID_FROM_TRDID( trdid )   (ltid_t)(trdid & 0x0000FFFF) 
#define CXY_FROM_TRDID( trdid )    (cxy_t)(trdid >> 16)
#define TRDID( cxy , ltid )        (trdid_t)((cxy << 16) | ltid )

/*********************************************************************************************
 * This structure defines an array of extended pointers on the open file descriptors
 * for a given process. We use an extended pointer because the open file descriptor
 * is always stored in the same cluster as the inode associated to the file.
 * A free entry in this array contains the XPTR_NULL value.
 * The array size is defined by a the CONFIG_PROCESS_FILE_MAX_NR parameter.
 * All modifications (open/close) must be done by the reference cluster, and reported
 ********************************************************************************************/

typedef struct fd_array_s
{
	remote_spinlock_t lock;             /*! lock protecting fd_array[] change.              */
    uint32_t          current;          /*! current number of open files                    */
    uint32_t          max;              /*! max number of open files (can be increased)     */
	xptr_t            array[CONFIG_PROCESS_FILE_MAX_NR];  
}
fd_array_t;

/*********************************************************************************************
 * This structure defines a process descriptor.
 * A process is identified by a unique PID (process identifier):
 * - The PID 16 LSB bits contain the LPID (Local Process Index)
 * - The PID 16 MSB bits contain the owner cluster CXY.
 * In each cluster, the process manager allocates LPID values for the process that are
 * allocated to this cluster.
 * The process descriptor for a PID process is replicated in all clusters containing
 * at least one thread of the PID process, with some restrictions:
 * - The list of registered vsegs, the page table (contained in vmm), and the fd_array[]
 *   are only complete in the reference process descriptor, other copies are read-only caches.
 * - The following fields are NOT defined in copies other than the reference:
 *   children_root , children_list , children_nr , sig_mgr
 ********************************************************************************************/

typedef struct process_s
{
	vmm_t            vmm;               /*! embedded virtual memory manager                 */

	fd_array_t       fd_array;          /*! embedded file descriptors array                 */

	xptr_t           vfs_root_xp;       /*! extende pointer on current file system root     */
	xptr_t           vfs_cwd_xp;        /*! extended pointer on current working directory   */
	xptr_t           vfs_bin_xp;        /*! extended pointer on associate .elf file         */

	spinlock_t       cd_lock;           /*! lock protecting working directory changes       */
 
	pid_t            pid;               /*! process identifier                              */
	pid_t            ppid;              /*! parent process identifier                       */
	bool_t           is_ref;            /*! this is the reference process if true           */
    xptr_t           ref_xp;            /*! extended pointer on reference process           */

	xlist_entry_t    children_root;     /*! root of the children process xlist              */
    uint32_t         children_nr;       /*! number of children processes                    */

	xlist_entry_t    brothers_list;     /*! member of list of children of same parent       */

    list_entry_t     local_list;        /*! member of list of process in same cluster       */

    xlist_entry_t    copies_list;       /*! member of list of copies of same process        */

	spinlock_t       th_lock;           /*! lock protecting th_tbl[] concurrent access      */
	uint32_t         th_nr;             /*! number of threads in this cluster               */

	struct thread_s * th_tbl[CONFIG_THREAD_MAX_PER_CLUSTER]; /*! pointers on local threads  */

    xlist_entry_t    sem_root;          /*! root of the process semaphores list             */
    uint32_t         sem_nr;            /*! number of semaphores                            */

	sig_mgr_t        sig_mgr;           /*! embedded signal manager TODO [AG]               */
}
process_t;

/*********************************************************************************************
 * This structure defines the minimal information required by the process_make_exec()
 * function to build a new reference process descriptor, and the associated main thread.
 ********************************************************************************************/

typedef struct exec_info_s
{
    pid_t              pid;            /*! process identifier                               */
    pid_t              ppid;           /*! parent process identifier                        */

    xptr_t             fd_array_xp;    /*! extended pointer on parent process fd_array      */

    xptr_t             vfs_root_xp;    /*! extended pointer on file system root             */
    xptr_t             vfs_cwd_xp;     /*! extended pointer on current working directory    */
    xptr_t             vfs_bin_xp;     /*! extended pointer on process .elf file            */

    char               path[256];      /*! pathname for  .elf file                          */

    char            ** args_pointers;  /*! physical base address of array of pointers       */
    char             * args_buf_base;  /*! physical base address of kernel args buffer      */
    uint32_t           args_nr;        /*! actual number of arguments                       */

    char            ** envs_pointers;  /*! physical base address of array of pointers       */
    char             * envs_buf_base;  /*! physical base address of kernel args buffer      */
    char             * envs_buf_free;  /*! physical address of first free slot in envs_buf  */
    uint32_t           envs_nr;        /*! actual number of environment variables           */
}
exec_info_t;

/*********************************************************************************************
 * This macro returns a pointer on the process descriptor for the calling thread.
 ********************************************************************************************/

#define CURRENT_PROCESS (hal_get_current_thread()->process)

/***************   Process Descriptor Operations    *****************************************/

/********************************************************************************************* 
 * This function allocates memory in local cluster for a process descriptor. 
 ********************************************************************************************* 
 * @ returns pointer on process descriptor if success / return NULL if failure
 ********************************************************************************************/
process_t * process_alloc();

/********************************************************************************************* 
 * This function releases memory in local cluster for a process descriptor. 
 ********************************************************************************************* 
 * @ process      : pointer on process descriptor to release.
 ********************************************************************************************/
void process_free( process_t * process );

/*********************************************************************************************
 * This function initializes the kernel process_zero descriptor (global variable) from
 * informations found in the boot_info.
 ********************************************************************************************* 
 * @ info      : pointer on local boot_info_t structure.
 ********************************************************************************************/
void process_zero_init( boot_info_t * info );

/********************************************************************************************* 
 * This function allocates memory and initialises the "process_init" descriptor and the
 * associated thread descriptor from information found in the "process_zero" descriptor.
 ********************************************************************************************* 
 * Any error gives a kernel panic.
 ********************************************************************************************/
void process_init_create();

/********************************************************************************************* 
 * This function intializes a new process descriptor, in the reference cluster.
 * The PID value must have been defined previously by the owner cluster manager.
 * The reference cluster can be different from the owner cluster. 
 * It set the pid / ppid / is_ref / pref / fields.
 * It registers this process descriptor in three lists:
 * - the children_list in the parent process descriptor.
 * - the local_list, rooted in the reference cluster manager.
 * - the copies_list, rooted in the owner cluster manager.
 * It reset the embedded structures such as the VMM or the file descriptor array.
 ********************************************************************************************* 
 * @ process      : [in] pointer on process descriptor to initialize.
 * @ pid          : [in] process identifier defined by owner cluster.
 * @ ppid         : [in] parent process identifier.
 ********************************************************************************************/
void process_reference_init( process_t * process,
                             pid_t       pid,
                             pid_t       ppid );

/********************************************************************************************* 
 * This function intializes a copy process descriptor, in the local cluster,
 * from informations defined in the reference remote process descriptor.
 ********************************************************************************************* 
 * @ process              : [in] local pointer on process descriptor to initialize.
 * @ reference_process_xp : [in] extended pointer on reference process descriptor.
 * @ return 0 if success / return ENOMEM if failure
 ********************************************************************************************/
error_t process_copy_init( process_t * local_process,
                           xptr_t      reference_process_xp );

/********************************************************************************************* 
 * This function releases all memory allocated for a process descriptor in the local cluster,
 * after releasing memory allocated for embedded substructures (fd_array, vmm, etc).
 * The local th_tbl[] array must be empty.
 ********************************************************************************************* 
 * @ process     : pointer on the process descriptor.
 ********************************************************************************************/
void process_destroy( process_t * process );

/********************************************************************************************* 
 * This function kills an user process in a given cluster.
 * It can be directly called in the reference cluster, or it can be called through the
 * PROCESS_KILL RPC.
 * - In a first loop, it set the THREAD_SIG_KILL signal to all threads of process.
 * - In a second loop, it wait, for each thread the reset of the THREAD_SIG_KILL signal
 *   by the scheduler, and completes the thread descriptor destruction. 
 ********************************************************************************************* 
 * @ process     : pointer on the process descriptor.
 ********************************************************************************************/
void process_kill( process_t * process );

/********************************************************************************************* 
 * This function returns a pointer on the local copy of a process identified by its PID.
 * If this local copy does not exist yet, it is dynamically created, from the reference 
 * process descriptor, registered in the global copies_list, and registered in the local_list.
 * This function is used by both thread_user_create() and thread_user_copy() functions.
 ********************************************************************************************* 
 * @ pid     : searched process identifier.
 * @ returns pointer on the local process descriptor if success / returns NULL if failure.
 ********************************************************************************************/
process_t * process_get_local_copy( pid_t pid );

/********************************************************************************************* 
 * This function builds a new reference process descriptor and associated main thread.
 * It is executed in the local cluster, that becomes both "owner" and "reference".
 * The new process descriptor and the main_thread are initialised using only informations 
 * found in the local exec_info structure, that must be build by the caller.
 * - It can be called by the process_init_create() function to build the "init" process.
 * - It can be called directly by the sys_exec() function in case of local exec.
 * - It can be called through the rpc_process_exec_server() function in case of remote exec.
 ********************************************************************************************* 
 * @ exec_info   : [in]  pointer on the exec_info structure.
 * @ return 0 if success / return non-zero if error.
 ********************************************************************************************/
error_t process_make_exec( exec_info_t * exec_info );


/********************   Signal Management Operations   **************************************/ 

/********************************************************************************************* 
 * This function TODO [AG]
 ********************************************************************************************/
void process_signal_handler( process_t * process );


/********************   File Management Operations   ****************************************/ 

/*********************************************************************************************
 * This function reset the file descriptor array for a given process: no open file.
 ********************************************************************************************* 
 * @ process  : pointer on the local process descriptor.
 ********************************************************************************************/
void process_fd_init( process_t * process );

/*********************************************************************************************
 * This function closes all open files and releases all file descriptors.
 ********************************************************************************************* 
 * @ process  : pointer on the local process descriptor.
 ********************************************************************************************/
void process_fd_destroy( process_t * process );

/*********************************************************************************************
 * This function checks the number of open files for a given process. 
 ********************************************************************************************* 
 * @ process  : pointer on the local process descriptor.
 * @ returns true if file descriptor array full.
 ********************************************************************************************/
bool_t process_fd_array_full( process_t * process );

/*********************************************************************************************
 * These functions allocates a slot in the fd_array for a given process,
 * and returns the file descriptor index.
 ********************************************************************************************* 
 * @ process  : pointer on the local process descriptor.
 * @ file     : extended pointer on the file descriptor.
 * @ fd       : [out] file descriptor index
 * @ return 0 if success / return EMFILE if array full.
 ********************************************************************************************/
error_t process_fd_allocate( process_t * process, 
                             xptr_t      file_xp, 
                             uint32_t  * fd );

/*********************************************************************************************
 * This function releases an existing file descriptor from the process fd_array. 
 ********************************************************************************************* 
 * @ process  : pointer on the local process descriptor.
 * @ return 0 if success / return EBADF if illegal file descriptor.
 ********************************************************************************************/
error_t process_fd_release( process_t * process,
                            uint32_t    fd );

/*********************************************************************************************
 * This function copies all non-zero entries from a local <src> fd_array,
 * embedded in a process descriptor, to another local <dst_xp> fd_array, embedded
 * in another process descriptor. 
 * It takes the remote lock protecting the <src> fd_array during the copy.
 * For each involved file descriptor, the refcount is incremented. 
 ********************************************************************************************* 
 * @ dst   : pointer on the destination fd_array_t.
 * @ src   : pointer on the source fd_array_t.
 ********************************************************************************************/
void process_fd_local_copy( fd_array_t * dst,
                            fd_array_t * src );

/*********************************************************************************************
 * This function copies all non-zero entries from a remote <src_xp> fd_array,
 * embedded in a process descriptor, to another remote <dst_xp> fd_array, embedded
 * in another process descriptor. The calling thread can be running in any cluster.
 * It takes the remote lock protecting the <src_xp> fd_array during the copy.
 * For each involved file descriptor, the refcount is incremented. 
 ********************************************************************************************* 
 * @ dst_xp   : extended pointer on the destination fd_array_t.
 * @ src_xp   : extended pointer on the source fd_array_t.
 ********************************************************************************************/
void process_fd_remote_copy( xptr_t dst_xp,
                             xptr_t src_xp );


/********************   Thread Related Operations   *****************************************/ 

/*********************************************************************************************
 * This function registers a new thread in the local process descriptor.
 * It checks that there is an available slot in the local th_tbl[] array,
 * allocates a new LTID, and registers the new thread in the th_tbl[].
 * WARNING : the lock protecting the th_tbl[] must be taken by the caller.
 ********************************************************************************************* 
 * @ process  : pointer on the local process descriptor.
 * @ thread   : pointer on new thread to be registered.
 * @ trdid    : [out] address of buffer for allocated trdid.
 * @ returns 0 if success / returns non zero if no slot available.
 ********************************************************************************************/
error_t process_register_thread( process_t       * process,
                                 struct thread_s * thread,
                                 trdid_t         * trdid );

/*********************************************************************************************
 * This function removes a thread registration from the local process descriptor.
 * WARNING : the lock protecting the th_tbl[] must be taken by the caller.
 ********************************************************************************************* 
 * @ thread   : local pointer on thread to be removed.
 ********************************************************************************************/
void process_remove_thread( struct thread_s * thread );



#endif	/* _PROCESS_H_ */
