/*
 * printk.h - Kernel Log & debug messages API definition.
 * 
 * authors  Alain Greiner (2016)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

///////////////////////////////////////////////////////////////////////////////////
// The printk.c and printk.h files define the functions used by the kernel
// to display messages on a text terminal. 
// Two access modes are supported:
// - The printk() function displays kernel messages on the kernel terminal TXT0,
//   using a busy waiting policy: It calls directly the relevant TXT driver, 
//   after taking the TXT0 chdev lock for exclusive access to the TXT0 terminal.
// - The user_printk() function displays messages on the calling thread private
//   terminal, using a descheduling policy: it register the request in the selected
//   TXT chdev waiting queue and deschedule. The calling thread is reactivated by 
//   the IRQ signaling completion.
// Both functions use the generic TXT device to call the proper implementation
// dependant TXT driver.
// Finally these files define a set of conditionnal trace <***_dmsg> for debug.
///////////////////////////////////////////////////////////////////////////////////

#ifndef _PRINTK_H
#define _PRINTK_H

#include <hal_types.h>
#include <stdarg.h>


/**********************************************************************************
 * This function build a formated string.
 * The supported formats are defined below :
 *   %c : single character
 *   %d : signed decimal 32 bits integer
 *   %u : unsigned decimal 32 bits integer
 *   %x : hexadecimal 32 bits integer
 *   %l : hexadecimal 64 bits integer
 *   %s : NUL terminated character string
 **********************************************************************************
 * @ string     : pointer on target buffer (allocated by caller).
 * @ length     : target buffer length (number of bytes).
 * @ format     : format respecting the printf syntax.
 * @ returns the string length (including NUL) if success / return -1 if error.
 *********************************************************************************/
uint32_t snprintf( char     * string,
                   uint32_t   length,
                   char     * format, ... );

/**********************************************************************************
 * This function displays a formated string on the kernel terminal TXT0,
 * using a busy waiting policy: It calls directly the relevant TXT driver, 
 * after taking the TXT0 chdev lock for exclusive access to the TXT0 terminal.
 **********************************************************************************
 * @ format     : formated string.
 *********************************************************************************/
void         printk( char* format, ... );

/**********************************************************************************
 * Display a formated string on the calling thread private terminal, using a 
 * descheduling policy: it register the request in the selected TXT chdev waiting
 * queue and deschedule. IT is reactivated by the IRQ signaling completion.
 * Not fully implemented yet ( use TXT0 in deschedling mode ).
 **********************************************************************************
 * @ format     : formated string.
 *********************************************************************************/
void         user_printk( char* format, ... );

/**********************************************************************************
 * This function displaya "PANIC" message and force the calling core in
 * sleeping mode if a Boolean condition is false.
 * These functions are actually used to debug the kernel...
 **********************************************************************************
 * @ condition     : condition that must be true.
 * @ function_name : name of the calling function.
 * @ string        : error message if condition is false. 
 *********************************************************************************/
inline void assert( bool_t       condition,
                    const char * function_name,
                    char       * string );

///////////////////////////////////////////////////////////////////////////////////
//       Conditionnal debug macros
///////////////////////////////////////////////////////////////////////////////////

#if CONFIG_CONTEXT_DEBUG
#define context_dmsg(...)   printk(__VA_ARGS__)
#else
#define context_dmsg(...)
#endif

#if CONFIG_CORE_DEBUG
#define core_dmsg(...)   printk(__VA_ARGS__)
#else
#define core_dmsg(...)
#endif

#if CONFIG_DQDT_DEBUG
#define dma_dmsg(...)   printk(__VA_ARGS__)
#else
#define dma_dmsg(...)
#endif

#if CONFIG_DQDT_DEBUG
#define dqdt_dmsg(...)   printk(__VA_ARGS__)
#else
#define dqdt_dmsg(...)
#endif

#if CONFIG_ELF_DEBUG
#define elf_dmsg(...)   printk(__VA_ARGS__)
#else
#define elf_dmsg(...)
#endif

#if CONFIG_EXEC_DEBUG
#define exec_dmsg(...)   printk(__VA_ARGS__)
#else
#define exec_dmsg(...)
#endif

#if CONFIG_FBF_DEBUG
#define fbf_dmsg(...)   printk(__VA_ARGS__)
#else
#define fbf_dmsg(...)
#endif

#if CONFIG_FORK_DEBUG
#define fork_dmsg(...)   printk(__VA_ARGS__)
#else
#define fork_dmsg(...)
#endif

#if CONFIG_ICU_DEBUG
#define icu_dmsg(...)   printk(__VA_ARGS__)
#else
#define icu_dmsg(...)
#endif

#if CONFIG_IOC_DEBUG
#define ioc_dmsg(...)   printk(__VA_ARGS__)
#else
#define ioc_dmsg(...)
#endif

#if CONFIG_KCM_DEBUG
#define kcm_dmsg(...) printk(__VA_ARGS__)
#else
#define kcm_dmsg(...)
#endif

#if CONFIG_KHM_DEBUG
#define khm_dmsg(...) printk(__VA_ARGS__)
#else
#define khm_dmsg(...)
#endif

#if CONFIG_KINIT_DEBUG
#define kinit_dmsg(...) printk(__VA_ARGS__)
#else
#define kinit_dmsg(...)
#endif

#if CONFIG_KMEM_DEBUG
#define kmem_dmsg(...) printk(__VA_ARGS__)
#else
#define kmem_dmsg(...)
#endif

#if CONFIG_MAPPER_DEBUG
#define mapper_dmsg(...)   printk(__VA_ARGS__)
#else
#define mapper_dmsg(...)
#endif

#if CONFIG_MMC_DEBUG
#define mmc_dmsg(...)   printk(__VA_ARGS__)
#else
#define mmc_dmsg(...)
#endif

#if CONFIG_NIC_DEBUG
#define nic_dmsg(...)   printk(__VA_ARGS__)
#else
#define nic_dmsg(...)
#endif

#if CONFIG_PIC_DEBUG
#define pic_dmsg(...)   printk(__VA_ARGS__)
#else
#define pic_dmsg(...)
#endif

#if CONFIG_PPM_DEBUG
#define ppm_dmsg(...)   printk(__VA_ARGS__)
#else
#define ppm_dmsg(...)
#endif

#if CONFIG_PROCESS_DEBUG
#define process_dmsg(...)   printk(__VA_ARGS__)
#else
#define process_dmsg(...)
#endif

#if CONFIG_RPC_DEBUG
#define rpc_dmsg(...)   printk(__VA_ARGS__)
#else
#define rpc_dmsg(...)
#endif

#if CONFIG_SCHED_DEBUG
#define sched_dmsg(...)   printk(__VA_ARGS__)
#else
#define sched_dmsg(...)
#endif

#if CONFIG_SIGNAL_DEBUG
#define signal_dmsg(...)   printk(__VA_ARGS__)
#else
#define signal_dmsg(...)
#endif

#if CONFIG_SYSCALL_DEBUG
#define syscall_dmsg(...)   printk(__VA_ARGS__)
#else
#define syscall_dmsg(...)
#endif

#if CONFIG_THREAD_DEBUG
#define thread_dmsg(...)   printk(__VA_ARGS__)
#else
#define thread_dmsg(...)
#endif

#if CONFIG_TXT_DEBUG
#define txt_dmsg(...)   printk(__VA_ARGS__)
#else
#define txt_dmsg(...)
#endif

#if CONFIG_VFS_DEBUG
#define vfs_dmsg(...)   printk(__VA_ARGS__)
#else
#define vfs_dmsg(...)
#endif

#if CONFIG_VMM_DEBUG
#define vmm_dmsg(...)   printk(__VA_ARGS__)
#else
#define vmm_dmsg(...)
#endif


#endif  // _PRINTK_H

// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

