/*
 * sys_exec.c - Kernel function implementing the "exec" system call.
 * 
 * Authors   Alain Greiner (2016,2017)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <kernel_config.h>
#include <hal_types.h>
#include <hal_uspace.h>
#include <errno.h>
#include <printk.h>
#include <core.h>
#include <vfs.h>
#include <cluster.h>
#include <process.h>
#include <thread.h>
#include <vmm.h>
#include <ppm.h>
#include <rpc.h>


////////////////////////////////////////////////i////////////////////////////////////// 
// This static function is called by the sys_exec() function to register the .elf
// pathname in the exec_info  structure, from a string stored in user space.
////////////////////////////////////////////////i////////////////////////////////////// 
// @ exec_info : pointer on the exec_info structure.
// @ pathname  : string containing the path to the .elf file (user space).
// return 0 if success / non-zero if one string too long, or too many strings. 
///////////////////////////////////////////////////////////////////////////////////////
static error_t process_exec_get_path( exec_info_t  * exec_info,
                                      char         * pathname )
{
    uint32_t length;

    // get string length
	length = hal_strlen_from_uspace( pathname );

	if( length >= CONFIG_VFS_MAX_PATH_LENGTH ) return EINVAL;

    // copy string to exec_info
    hal_copy_from_uspace( &exec_info->path[0] , pathname , length );

    return 0;
}

////////////////////////////////////////////////i////////////////////////////////////// 
// This static function is called twice by the sys_exec() function :
// - to register the main() arguments (args) in the exec_info structure.
// - to register the environment variables (envs) in the exec_info structure.
// In both cases the input is an array of string pointers in user space,
// and a set of strings in user space. 
// We allocate one physical page to store a kernel copy of the array of pointers,
// we allocate one or several physical pages to store the strings themselve,
// and register these buffers and the number of strings in the exec_info structure.
// The max number of strings is 1024 (for both args and envs). The numbers of pages 
// to store the (args) and (envs) strings are configuration parameters.
/////////////////////////////////////////////////////////////////////////////////////// 
// @ exec_info : pointer on the exec_info structure.
// @ is_args   : true if called for (args) / false if called for (envs).
// @ pointers  : array of pointers on the strings (in user space).
// @ return 0 if success / non-zero if too many strings or no more memory. 
///////////////////////////////////////////////////////////////////////////////////////
static error_t process_exec_get_strings( exec_info_t  * exec_info,
                                         bool_t         is_args,
			                             char        ** u_pointers )
{
    uint32_t     index;       // string index
    uint32_t     found_null;  // NULL pointer found in array of pointers
    uint32_t     length;      // string length
    kmem_req_t   req;         // kmem request
    page_t     * page;        // page descriptor
    uint32_t     order;       // ln2( number of pages to store strings )
    char      ** k_pointers;  // base of kernel array of pointers
    char       * k_buf_ptr;   // pointer on first empty slot in kernel strings buffer
    char       * k_buf_base;  // base address of the kernel strings buffer

    // compute ln2( number of pages for kernel strings buffer )
    if( is_args ) order = bits_log2( CONFIG_VMM_ARGS_SIZE );
    else          order = bits_log2( CONFIG_VMM_ENVS_SIZE );

	req.type   = KMEM_PAGE;
	req.flags  = AF_KERNEL | AF_ZERO;

    // allocate one physical page for kernel array of pointers
    req.type   = 0;
    page       = kmem_alloc( &req );

    if( page == NULL ) return ENOMEM;

    k_pointers = ppm_page2base( page );
    
    // allocate several physical pages to store the strings themselve
    req.type   = order;
    page       = kmem_alloc( &req );

    if( page == NULL ) return ENOMEM;

    k_buf_base = ppm_page2base( page );
    
    // copy the array of pointers to kernel buffer 
    hal_copy_from_uspace( k_pointers, 
                          u_pointers,
                          CONFIG_PPM_PAGE_SIZE );

    // scan kernel array of pointers to copy the strings
    found_null = 0;
    k_buf_ptr  = k_buf_base;
    for( index = 0 ; index < 1024 ; index++ )
    {
        if( k_pointers[index] == NULL ) 
        {
            found_null = 1;
            break;
        }

        // compute string length
	    length = hal_strlen_from_uspace( k_pointers[index] );
 
        // copy the user string to kernel buffer
        hal_copy_from_uspace( k_buf_ptr,
                              k_pointers[index],
                              length );

        // update k_pointer[index] entry
        k_pointers[index] = k_buf_ptr;

        // increment pointer on kernel strings buffer
        k_buf_ptr += (length + 1);
    }

    // update into exec_info structure 
    if( found_null && is_args )
    {
        exec_info->args_pointers  =  k_pointers;
        exec_info->args_buf_base  =  k_buf_base;
        exec_info->args_nr        =  index;
    }
    else if( found_null && !is_args )
    {
        exec_info->envs_pointers  =  k_pointers;
        exec_info->envs_buf_base  =  k_buf_base;
        exec_info->envs_buf_free  =  k_buf_ptr;
        exec_info->envs_nr        =  index;
    }
    else 
    {
        return EINVAL;
    }

    return 0;
} // end process_exec_get_strings()

/////////////////////////////////////////////////////////////////////////////////////////
// Implementation note:
// This function build an exec_info_t structure containing all informations
// required to create the new process descriptor and the associated thread.
// It calls the static process_exec_get_path() and process_exec_get_strings() functions
// to copy the .elf pathname, the main() arguments and the environment variables from
// user buffers to the exec_info_t structure, and call the process_make_exec() function.
/////////////////////////////////////////////////////////////////////////////////////////
int sys_exec( char  * filename,     // .elf file pathname
              char ** args,         // process arguments
              char ** envs )        // environment variables
{
    exec_info_t  exec_info;         // structure to pass to process_make_exec()
	error_t      error;
    paddr_t      paddr;

    thread_t   * this    = CURRENT_THREAD;
	process_t  * process = this->process;

    // check argument fileme
    error = vmm_v2p_translate( false , filename , &paddr );

	if( error )
    {
        printk("\n[ERROR] in %s : filename unmapped\n", __FUNCTION__ );
        this->errno = EINVAL;
        return -1;
    }

    // check argument fileme
    error = vmm_v2p_translate( false , args , &paddr );

	if( error )
    {
        printk("\n[ERROR] in %s : args unmapped\n", __FUNCTION__ );
        this->errno = EINVAL;
        return -1;
    }

    // check argument fileme
    error = vmm_v2p_translate( false , envs , &paddr );

	if( error )
    {
        printk("\n[ERROR] in %s : envs unmapped\n", __FUNCTION__ );
        this->errno = EINVAL;
        return -1;
    }

    // compute client_cxy (local cluster) and server_cxy (target cluster)
    cxy_t     cxy_server = CXY_FROM_PID( process->pid );
    cxy_t     cxy_client = local_cxy;
    bool_t    is_local   = (cxy_server == cxy_client);

    exec_dmsg("\n[INFO] %s starts for process %x on core %d in cluster %x"
                 " / target_cluster = %x / cycle %d\n",
                 __FUNCTION__, process->pid , CURRENT_CORE->lid, 
                 cxy_client, cxy_server, hal_time_stamp());

    // initialize exec_info structure 
    exec_info.pid         = process->pid;
    exec_info.ppid        = process->ppid;
    exec_info.fd_array_xp = XPTR( local_cxy , &process->fd_array );
    exec_info.vfs_root_xp = process->vfs_root_xp;
    exec_info.vfs_cwd_xp  = process->vfs_cwd_xp;
    exec_info.vfs_bin_xp  = process->vfs_bin_xp;

	// check pathname and store it in exec_info structure
	error = process_exec_get_path( &exec_info , filename );

	if ( error )
    {
        printk("\n[ERROR] in %s : elf pathname too long\n", __FUNCTION__ );
        this->errno = error;
        return -1;
    }

	// check and store args in exec_info structure
	error = process_exec_get_strings( &exec_info , true , args );

	if( error )  
    {
        printk("\n[ERROR] in %s : cannot access args\n", __FUNCTION__ );
        this->errno = error;
        return -1;
    }
	
	// check and store envs in exec_info structure
	error = process_exec_get_strings( &exec_info , false , envs );

	if( error )
    {
        printk("\n[ERROR] in %s : cannot access envs\n", __FUNCTION__ );
        this->errno = error;
        return -1;
    }
	
    exec_dmsg("\n[INFO] %s starts exec for process %x at cycle %d\n",
              __FUNCTION__, process->pid, hal_time_stamp() );

    if( is_local )  error = process_make_exec( &exec_info );
    else            rpc_process_exec_client( cxy_server , &exec_info , &error );

    if( error )
    {
        printk("\n[ERROR] in %s : cannot create new process %x\n",
               __FUNCTION__ , process->pid );
        this->errno = error;
        return -1;
    }

    exec_dmsg("\n[INFO] %s completes exec for process %x at cycle %d\n",
              __FUNCTION__, process->pid , hal_time_stamp() );

    // delete the calling thread an process
    thread_kill( CURRENT_THREAD );
    process_kill( CURRENT_THREAD->process );

    return 0;

} // end sys_exec()

