/*
 * hal_exception.c - implementation of exception handler for TSAR-MIPS32.
 * 
 * Author   Alain Greiner (2016, 2017)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_types.h>
#include <hal_irqmask.h>
#include <hal_exception.h>
#include <thread.h>
#include <printk.h>
#include <vmm.h>
#include <errno.h>
#include <scheduler.h>
#include <core.h>
#include <signal.h>
#include <syscalls.h>
#include <do_exception.h>
#include <remote_spinlock.h>
#include <mips32_uzone.h>

//////////////////////////////////////////////////////////////////////////////////////////
//  Extern global variables
//////////////////////////////////////////////////////////////////////////////////////////

extern remote_spinlock_t  exception_lock;  // allocated in the do_exception.c file.

//////////////////////////////////////////////////////////////////////////////////////////
// This enum defines the relevant values for XCODE field in mips32 CP0_CR register.
//////////////////////////////////////////////////////////////////////////////////////////

typedef enum
{
    XCODE_ADEL = 0x4,        // Illegal address for data load 
    XCODE_ADES = 0x5,        // Illegal address for data store
    XCODE_IBE  = 0x6,        // Instruction MMU exception       (can be NON-FATAL)
    XCODE_DBE  = 0x7,        // Data MMU exception              (can be NON-FATAL)
    XCODE_RI   = 0xA,        // Reserved instruction exception
    XCODE_CPU  = 0xB,        // Coprocessor unusable exception  (can be NON-FATAl)
    XCODE_OVR  = 0xC,        // Arithmetic Overflow exception
}
xcode_values_t;

///////////////////////////////////////
void hal_do_exception( thread_t * this, 
                       reg_t    * regs_tbl )
{
	error_t             error;
	uint32_t            excCode;                  // 4 bits XCODE from CP0_CR

    // get 4 bits XCODE from CP0_CR register
	excCode        = (regs_tbl[UZ_CR] >> 2) & 0xF;

	switch(excCode)
	{
        case XCODE_DBE:     // can be non fatal
	    case XCODE_IBE:     // can be non fatal
        {
            // call generic excepton handler for a MMU exception
		    error = do_exception( this , true );
        }
		break;

	    case XCODE_CPU:    // can be non fatal
        {
            if( ((regs_tbl[UZ_CR] >> 28) & 0x3) == 1 )  // unavailable FPU
            {
                // call generic excepton handler for a FPU exception
                error = do_exception( this , false );
            }
            else
            {
		        printk("\n[ERROR] in thread %x / unsupported coprocessor type\n",
		               this->trdid );
		        error = EXCP_USER_ERROR;
            }
        }
		break;

        case XCODE_OVR:   // user fatal error
        {
            printk("\n[ERROR] in thread %x / arithmetic overflow\n",
		           this->trdid );
		    error = EXCP_USER_ERROR;
        }
	    break;

        case XCODE_RI:   // user fatal error
        {
            printk("\n[ERROR] in thread %x / Illegal Codop\n",
		           this->trdid );
		    error = EXCP_USER_ERROR;
        }
        break;

        case XCODE_ADEL: // user fatal error

        case XCODE_ADES:
        {
            printk("\n[ERROR] in thread %x / illegal address\n",
		           this->trdid );
		    error = EXCP_USER_ERROR;
        }
		break;

        default:
        {
            printk("\n[PANIC] in %s for thread %x / illegal XCODE value = %x\n",
		           __FUNCTION__ , this->trdid , excCode );
		    error = EXCP_USER_ERROR;
        }
	}
    
    // analyse error code
	if( error == EXCP_USER_ERROR )      //  user error => kill the user process and return
	{
        hal_exception_dump( this , regs_tbl );
        sys_kill( this->process->pid , SIGKILL );
	}
    else if( error == EXCP_KERNEL_PANIC )   // kernel error => kernel panic
    {
        hal_exception_dump( this , regs_tbl );
        hal_core_sleep();
    }
}  // end hal_do_exception()

/////////////////////////////////////////
void hal_exception_dump( thread_t * this,
                         reg_t    * regs_tbl )
{
    // take the exception_lock located in io_cluster 
    remote_spinlock_lock( XPTR( LOCAL_CLUSTER->io_cxy , &exception_lock ) );

    if( this->type == THREAD_USER )
    printk("\n================= USER ERROR =======================================\n");
    else
    printk("\n================= KERNEL PANIC =====================================\n");

	printk("  thread type = %s / trdid = %x / pid %x / core[%x,%d]\n"
           "  local locks = %d / remote locks = %d / blocked_vector = %X\n\n",
           thread_type_str(this->type), this->trdid, this->process->pid, local_cxy,
           this->core->lid, this->local_locks, this->remote_locks, this->blocked );

	printk("CR    %X  EPC   %X  SR    %X  SP     %X\n",
		   regs_tbl[UZ_CR], regs_tbl[UZ_EPC], regs_tbl[UZ_SR], regs_tbl[UZ_SP]);

    printk("at_1  %X  v0_2  %X  v1_3  %X  a0_4   %X  a1_5   %X\n",
   	       regs_tbl[UZ_AT], regs_tbl[UZ_V0], regs_tbl[UZ_V1], regs_tbl[UZ_A0], regs_tbl[UZ_A1]);

    printk("a2_6  %X  a3_7  %X  t0_8  %X  t1_9   %X  t2_10  %X\n",
		   regs_tbl[UZ_A2],regs_tbl[UZ_A3],regs_tbl[UZ_T0],regs_tbl[UZ_T1],regs_tbl[UZ_T2]);
  
    printk("t3_11 %X  t4_12 %X  t5_13 %X  t6_14  %X  t7_15  %X\n",
		   regs_tbl[UZ_T3],regs_tbl[UZ_T4],regs_tbl[UZ_T5],regs_tbl[UZ_T6],regs_tbl[UZ_T7]);

    printk("t8_24 %X  t9_25 %X  gp_28 %X  c0_hi  %X  c0_lo  %X\n",
		   regs_tbl[UZ_T8],regs_tbl[UZ_T9],regs_tbl[UZ_GP],regs_tbl[UZ_HI],regs_tbl[UZ_LO]);

    printk("s0_16 %X  s1_17 %X  s2_18 %X  s3_19  %X  s4_20  %X\n",
		   regs_tbl[UZ_S0],regs_tbl[UZ_S1],regs_tbl[UZ_S2],regs_tbl[UZ_S3],regs_tbl[UZ_S4]);
  
    printk("s5_21 %X  s6_22 %X  s7_23 %X  s8_30  %X  ra_31  %X\n",
	       regs_tbl[UZ_S5],regs_tbl[UZ_S6],regs_tbl[UZ_S7],regs_tbl[UZ_S8],regs_tbl[UZ_RA]);

    // release the exception_lock
    remote_spinlock_unlock( XPTR( LOCAL_CLUSTER->io_cxy , &exception_lock ) );

}  // end hal_exception_dump()

