/*
 * process.c - process related management
 *
 * Authors  Ghassan Almaless (2008,2009,2010,2011,2012)
 *          Mohamed Lamine Karaoui (2015)
 *          Alain Greiner (2016,2017)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH..
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <kernel_config.h>
#include <hal_types.h>
#include <hal_remote.h>
#include <hal_uspace.h>
#include <errno.h>
#include <printk.h>
#include <memcpy.h>
#include <bits.h>
#include <kmem.h>
#include <page.h>
#include <vmm.h>
#include <vfs.h>
#include <core.h>
#include <thread.h>
#include <list.h>
#include <scheduler.h>
#include <remote_spinlock.h>
#include <dqdt.h>
#include <cluster.h>
#include <ppm.h>
#include <boot_info.h>
#include <process.h>
#include <elf.h>
#include <syscalls.h>

//////////////////////////////////////////////////////////////////////////////////////////
// Extern global variables
//////////////////////////////////////////////////////////////////////////////////////////

extern process_t process_zero;

//////////////////////////////////////////////////////////////////////////////////////////
// Process initialisation related functions
//////////////////////////////////////////////////////////////////////////////////////////

///////////////////////////
process_t * process_alloc()
{
	kmem_req_t   req;

    req.type  = KMEM_PROCESS;
	req.size  = sizeof(process_t);
	req.flags = AF_KERNEL;

    return (process_t *)kmem_alloc( &req );
}

////////////////////////////////////////
void process_free( process_t * process )
{
    kmem_req_t  req;

	req.type = KMEM_PROCESS;
	req.ptr  = process;
	kmem_free( &req );
}

/////////////////////////////////////////////////
void process_reference_init( process_t * process,
                             pid_t       pid,
                             xptr_t      parent_xp )
{
    cxy_t       parent_cxy;
    process_t * parent_ptr;
    pid_t       parent_pid;

    process_dmsg("\n[INFO] %s : enters for process %x in cluster %x\n",
                 __FUNCTION__ , pid , local_cxy );

    // get parent process cluster, local pointer, and pid
    // for all processes other than process_zero
    if( process == &process_zero )
    {
        assert( (pid == 0) , __FUNCTION__ , "process_zero must have PID = 0\n");

        parent_cxy = 0;
        parent_ptr = NULL;
        parent_pid = 0;      // process_zero is its own parent...
    }
    else
    {
        assert( (parent_xp != XPTR_NULL) , __FUNCTION__ , "parent_xp cannot be NULL\n");

        parent_cxy = GET_CXY( parent_xp );
        parent_ptr = (process_t *)GET_PTR( parent_xp );
        parent_pid = hal_remote_lw( XPTR( parent_cxy , &parent_ptr->pid ) );
    }

    // initialize PID and PPID
	process->pid   = pid;
    process->ppid  = parent_pid;

    // reset reference process vmm (not for kernel process)
    if( pid ) vmm_init( process );

    // reset reference process file descriptors array
	process_fd_init( process );

    // reset reference process files structures and cwd_lock
	process->vfs_root_xp     = XPTR_NULL;
	process->vfs_bin_xp      = XPTR_NULL;
	process->vfs_cwd_xp      = XPTR_NULL;
    remote_rwlock_init( XPTR( local_cxy , &process->cwd_lock ) );

    // reset children list root
    xlist_root_init( XPTR( local_cxy , &process->children_root ) );
	process->children_nr     = 0;

    // reset semaphore / mutex / barrier / condvar list roots
    xlist_root_init( XPTR( local_cxy , &process->sem_root ) );
    xlist_root_init( XPTR( local_cxy , &process->mutex_root ) );
    xlist_root_init( XPTR( local_cxy , &process->barrier_root ) );
    xlist_root_init( XPTR( local_cxy , &process->condvar_root ) );
    remote_spinlock_init( XPTR( local_cxy , &process->sync_lock ) );

    // register new process in the parent children list (not for kernel process)
    if( pid )
    {
        xptr_t entry = XPTR( local_cxy  , &process->brothers_list );
        xptr_t root  = XPTR( parent_cxy , &parent_ptr->children_root );
        xlist_add_first( root , entry );
    }

    // reset th_tbl[] array as empty
    uint32_t i;
    for( i = 0 ; i < CONFIG_THREAD_MAX_PER_CLUSTER ; i++ )
	{
        process->th_tbl[i] = NULL;
    }
    process->th_nr  = 0;
    spinlock_init( &process->th_lock );

    // set ref_xp field
    process->ref_xp = XPTR( local_cxy , process );

    // register new process descriptor in local cluster manager local_list
    cluster_process_local_link( process );

    // register new process descriptor in owner cluster manager copies_list
    cluster_process_copies_link( process );

    // initialize signal manager TODO [AG]

	hal_fence();

    process_dmsg("\n[INFO] %s : exit for process %x in cluster %x\n",
                 __FUNCTION__ , pid );

}  // process_reference init()

/////////////////////////////////////////////////////
error_t process_copy_init( process_t * local_process,
                           xptr_t      reference_process_xp )
{
    // get reference process cluster and local pointer
    cxy_t       ref_cxy = GET_CXY( reference_process_xp );
    process_t * ref_ptr = (process_t *)GET_PTR( reference_process_xp );

    // reset local process vmm
    vmm_init( local_process );

    // reset process file descriptors array
	process_fd_init( local_process );

    // reset vfs_root_xp / vfs_bin_xp / vfs_cwd_xp fields
    local_process->vfs_root_xp = hal_remote_lwd( XPTR( ref_cxy , &ref_ptr->vfs_root_xp ) );
    local_process->vfs_bin_xp  = hal_remote_lwd( XPTR( ref_cxy , &ref_ptr->vfs_bin_xp ) );
    local_process->vfs_cwd_xp  = XPTR_NULL;

    // set the pid, ppid, ref_xp fields
    local_process->pid    = hal_remote_lw( XPTR( ref_cxy , &ref_ptr->pid ) );
    local_process->ppid   = hal_remote_lw( XPTR( ref_cxy , &ref_ptr->ppid ) );
    local_process->ref_xp = reference_process_xp;

    process_dmsg("\n[INFO] %s : enter for process %x in cluster %x\n",
                 __FUNCTION__ , local_process->pid );

    // reset children list root (not used in a process descriptor copy)
    xlist_root_init( XPTR( local_cxy , &local_process->children_root ) );
    local_process->children_nr   = 0;

    // reset brothers list (not used in a process descriptor copy)
    xlist_entry_init( XPTR( local_cxy , &local_process->brothers_list ) );

    // reset semaphores list root (not used in a process descriptor copy)
    xlist_root_init( XPTR( local_cxy , &local_process->sem_root ) );
    xlist_root_init( XPTR( local_cxy , &local_process->mutex_root ) );
    xlist_root_init( XPTR( local_cxy , &local_process->barrier_root ) );
    xlist_root_init( XPTR( local_cxy , &local_process->condvar_root ) );

    // reset th_tbl[] array as empty
    uint32_t i;
    for( i = 0 ; i < CONFIG_THREAD_MAX_PER_CLUSTER ; i++ )
	{
        local_process->th_tbl[i] = NULL;
    }
    local_process->th_nr  = 0;
    spinlock_init( &local_process->th_lock );

    // register new process descriptor in local cluster manager local_list
    cluster_process_local_link( local_process );

    // register new process descriptor in owner cluster manager copies_list
    cluster_process_copies_link( local_process );

    // initialize signal manager TODO [AG]

	hal_fence();

    process_dmsg("\n[INFO] %s : exit for process %x in cluster %x\n",
                 __FUNCTION__ , local_process->pid );

    return 0;

} // end process_copy_init()

///////////////////////////////////////////
void process_destroy( process_t * process )
{
	if( process->th_nr != 0 )
    {
        panic("process %x in cluster %x has still active threads",
              process->pid , local_cxy );
    }

    // get local process manager pointer
    pmgr_t * pmgr = &LOCAL_CLUSTER->pmgr;

    // get the lock protecting the list of local process descriptors
    remote_spinlock_lock( XPTR( local_cxy , &pmgr->local_lock ) );

    // remove the process descriptor from local_list in local cluster manager
    xlist_unlink( XPTR( local_cxy , &process->local_list ) );

    // release the lock protecting the list of local process descriptors
    remote_spinlock_unlock( XPTR( local_cxy , &pmgr->local_lock ) );

    // get extended pointer on copies_lock in owner cluster manager
    cxy_t  owner_cxy    = CXY_FROM_PID( process->pid );
	lpid_t lpid         = LPID_FROM_PID( process->pid );
    xptr_t copies_lock  = XPTR( owner_cxy , &pmgr->copies_lock[lpid] );

    // remove the local process descriptor from copies_list
    remote_spinlock_lock( copies_lock );
    xlist_unlink( XPTR( local_cxy , &process->copies_list ) );
    remote_spinlock_unlock( copies_lock );

    // synchronize memory
	hal_fence();

    // From this point, the process descriptor is unreachable

    // close all open files and update dirty TODO [AG]

    // release signal manager TODO [AG]

    // Decrease refcount for bin file, root file and cwd file
	if( process->vfs_bin_xp  != XPTR_NULL ) vfs_file_count_down( process->vfs_bin_xp );
	if( process->vfs_root_xp != XPTR_NULL ) vfs_file_count_down( process->vfs_root_xp );
	if( process->vfs_cwd_xp  != XPTR_NULL ) vfs_file_count_down( process->vfs_cwd_xp );

    // Destroy VMM
    vmm_destroy( process );

	process_dmsg("\n[INFO] %s for pid %d / page_faults = %d\n",
	         __FUNCTION__ , process->pid, process->vmm.pgfault_nr );
}

////////////////////////////////////////
void process_kill( process_t * process )
{
    thread_t     * thread;    // pointer on current thead descriptor
    uint32_t       ltid;      // index in process th_tbl
    uint32_t       count;     // thread counter

    // get lock protecting th_tbl[]
    spinlock_lock( &process->th_lock );

    // first loop on threads to send the THREAD_SIG_KILL signal to all process threads
    // we use both "ltid" and "count" indexes, because it can exist "holes" in th_tbl
    for( ltid = 0 , count = 0  ;
         (ltid < CONFIG_THREAD_MAX_PER_CLUSTER) && (count < process->th_nr) ;
         ltid++ )
    {
        thread = process->th_tbl[ltid];

        if( thread != NULL )
        {
            thread_kill( thread );
            count++;
        }
    }

    volatile uint32_t ko;

    // second loop on threads to wait acknowledge from scheduler,
    // unlink thread from process and parent thread, and release thread descriptor
    for( ltid = 0 , count = 0  ;
         (ltid < CONFIG_THREAD_MAX_PER_CLUSTER) && (count < process->th_nr) ;
         ltid++ )
    {
        thread = process->th_tbl[ltid];

        if( thread != NULL )
        {
            // wait scheduler acknowledge
            do { ko = (thread->signals & THREAD_SIG_KILL); } while( ko );

            // unlink thread from brothers list if required
            if( (thread->flags & THREAD_FLAG_DETACHED) == 0 )
            xlist_unlink( XPTR( local_cxy , &thread->brothers_list ) );

            // unlink thread from process
            process_remove_thread( thread );

            // release memory for thread descriptor
            thread_destroy( thread );

            count++;
        }
    }

    // release lock protecting th_tbl[]
    spinlock_unlock( &process->th_lock );

    // release memory allocated for process descriptor
    process_destroy( process );
}

///////////////////////////////////////////////
process_t * process_get_local_copy( pid_t pid )
{
    error_t        error;
    process_t    * process_ptr;   // local pointer on process
    xptr_t         process_xp;    // extended pointer on process

    cluster_t * cluster = LOCAL_CLUSTER;

    // get lock protecting local list of processes
    remote_spinlock_lock( XPTR( local_cxy , &cluster->pmgr.local_lock ) );

    // scan the local list of process descriptors to find the process
    xptr_t  iter;
    bool_t  found = false;
    XLIST_FOREACH( XPTR( local_cxy , &cluster->pmgr.local_root ) , iter )
    {
        process_xp  = XLIST_ELEMENT( iter , process_t , local_list );
        process_ptr = (process_t *)GET_PTR( process_xp );
        if( process_ptr->pid == pid )
        {
            found = true;
            break;
        }
    }

    // release lock protecting local list of processes
    remote_spinlock_unlock( XPTR( local_cxy , &cluster->pmgr.local_lock ) );

    // allocate memory for a new local process descriptor
    // and initialise it from reference cluster if required
    if( !found )
    {
        // get extended pointer on reference process descriptor
        xptr_t ref_xp = cluster_get_reference_process_from_pid( pid );

        assert( (ref_xp != XPTR_NULL) , __FUNCTION__ , "illegal pid\n" );

        // allocate memory for local process descriptor
        process_ptr = process_alloc();
        if( process_ptr == NULL )  return NULL;

        // initialize local process descriptor copy
        error = process_copy_init( process_ptr , ref_xp );
        if( error ) return NULL;
    }

    return process_ptr;
}

//////////////////////////////////////////////////////////////////////////////////////////
// File descriptor array related functions
//////////////////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////
void process_fd_init( process_t * process )
{
    uint32_t fd;

    remote_spinlock_init( XPTR( local_cxy , &process->fd_array.lock ) );

    process->fd_array.current = 0;

    // initialize array
    for ( fd = 0 ; fd < CONFIG_PROCESS_FILE_MAX_NR ; fd++ )
    {
        process->fd_array.array[fd] = XPTR_NULL;
    }
}

//////////////////////////////
bool_t process_fd_array_full()
{
    // get extended pointer on reference process
    xptr_t ref_xp = CURRENT_THREAD->process->ref_xp;

    // get reference process cluster and local pointer
    process_t * ref_ptr = (process_t *)GET_PTR( ref_xp );
    cxy_t       ref_cxy = GET_CXY( ref_xp );

    // get number of open file descriptors from reference fd_array
    uint32_t current = hal_remote_lw( XPTR( ref_cxy , &ref_ptr->fd_array.current ) );

	return ( current >= CONFIG_PROCESS_FILE_MAX_NR );
}

/////////////////////////////////////////////////
error_t process_fd_register(  xptr_t     file_xp,
                              uint32_t * file_id )
{
    bool_t    found;
    uint32_t  id;
    xptr_t    xp;

    // get extended pointer on reference process
    xptr_t ref_xp = CURRENT_THREAD->process->ref_xp;

    // get reference process cluster and local pointer
    process_t * ref_ptr = (process_t *)GET_PTR( ref_xp );
    cxy_t       ref_cxy = GET_CXY( ref_xp );

    // take lock protecting reference fd_array
	remote_spinlock_lock( XPTR( ref_cxy , &ref_ptr->fd_array.lock ) );

    found   = false;

    for ( id = 0; id < CONFIG_PROCESS_FILE_MAX_NR ; id++ )
    {
        xp = hal_remote_lwd( XPTR( ref_cxy , &ref_ptr->fd_array.array[id] ) );
        if ( xp == XPTR_NULL )
        {
            found = true;
            hal_remote_swd( XPTR( ref_cxy , &ref_ptr->fd_array.array[id] ) , file_xp );
	        hal_remote_atomic_add( XPTR( ref_cxy , &ref_ptr->fd_array.current ) , 1 );
			*file_id = id;
            break;
        }
    }

    // release lock protecting reference fd_array
	remote_spinlock_unlock( XPTR( ref_cxy , &ref_ptr->fd_array.lock ) );

    if ( !found ) return EMFILE;
    else          return 0;
}

////////////////////////////////////////////////
xptr_t process_fd_get_xptr( process_t * process,
                            uint32_t    file_id )
{
    xptr_t  file_xp;

    // access local copy of process descriptor
    file_xp = process->fd_array.array[file_id];

    if( file_xp == XPTR_NULL )
    {
        // get reference process cluster and local pointer
        xptr_t      ref_xp  = process->ref_xp;
        cxy_t       ref_cxy = GET_CXY( ref_xp );
        process_t * ref_ptr = (process_t *)GET_PTR( ref_xp );

        // access reference process descriptor
        file_xp = hal_remote_lwd( XPTR( ref_cxy , &ref_ptr->fd_array.array[file_id] ) );

        // update local fd_array if found
        if( file_xp != XPTR_NULL )
        {
            process->fd_array.array[file_id] = file_xp;
        }
    }

    return file_xp;
}

///////////////////////////////////////////
void process_fd_remote_copy( xptr_t dst_xp,
                             xptr_t src_xp )
{
    uint32_t fd;
    xptr_t   entry;

    // get cluster and local pointer for src fd_array
    cxy_t        src_cxy = GET_CXY( src_xp );
    fd_array_t * src_ptr = (fd_array_t *)GET_PTR( src_xp );

    // get cluster and local pointer for dst fd_array
    cxy_t        dst_cxy = GET_CXY( dst_xp );
    fd_array_t * dst_ptr = (fd_array_t *)GET_PTR( dst_xp );

    // get the remote lock protecting the src fd_array
	remote_spinlock_lock( XPTR( src_cxy , &src_ptr->lock ) );

    // loop on all entries in source process fd_array
    for( fd = 0 ; fd < CONFIG_PROCESS_FILE_MAX_NR ; fd++ )
	{
		entry = (xptr_t)hal_remote_lwd( XPTR( src_cxy , &src_ptr->array[fd] ) );

		if( entry != XPTR_NULL )
		{
            // increment file descriptor ref count
            vfs_file_count_up( entry );

			// copy entry in destination process fd_array
			hal_remote_swd( XPTR( dst_cxy , &dst_ptr->array[fd] ) , entry );
		}
	}

    // release lock on source process fd_array
	remote_spinlock_unlock( XPTR( src_cxy , &src_ptr->lock ) );
}

////////////////////////////////////////////////////////////////////////////////////
//  Thread related functions
////////////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////
error_t process_register_thread( process_t * process,
                                 thread_t  * thread,
                                 trdid_t   * trdid )
{
    ltid_t   ltid;
    bool_t   found;

    assert( (process != NULL) , __FUNCTION__ , "process argument is NULL" );

    assert( (thread != NULL) , __FUNCTION__ , "thread argument is NULL" );

    // search a free slot in th_tbl[]
    found = false;
    for( ltid = 0 ; ltid < CONFIG_THREAD_MAX_PER_CLUSTER ; ltid++ )
    {
        if( process->th_tbl[ltid] == NULL )
        {
            found = true;
            break;
        }
    }

    if( found )
    {
        // register thread in th_tbl[]
        process->th_tbl[ltid] = thread;
        process->th_nr++;

        // returns trdid
        *trdid = TRDID( local_cxy , ltid );
    }

    return (found) ? 0 : ENOMEM;

}  // end process_register_thread()

///////////////////////////////////////////////
void process_remove_thread( thread_t * thread )
{
    assert( (thread != NULL) , __FUNCTION__ , "thread argument is NULL" );

    process_t * process = thread->process;

    // get thread local index
    ltid_t  ltid = LTID_FROM_TRDID( thread->trdid );

    // remove thread from th_tbl[]
    process->th_tbl[ltid] = NULL;
    process->th_nr--;

}  // process_remove_thread()

/////////////////////////////////////////////////////
error_t process_make_exec( exec_info_t  * exec_info )
{
    char           * path;                            // pathname to .elf file
    process_t      * process;                         // local pointer on new process
    pid_t            pid;                             // new process pid
    xptr_t           parent_xp;                       // extended pointer on parent process
    cxy_t            parent_cxy;
    process_t      * parent_ptr;
    uint32_t         parent_pid;
    thread_t       * thread;                          // pointer on new thread
    pthread_attr_t   attr;                            // main thread attributes
    core_t         * core;                            // pointer on selected core
    lid_t            lid;                             // selected core local index
	error_t          error;

	// get parent and .elf pathname from exec_info
	path      = exec_info->path;
    parent_xp = exec_info->parent_xp;

    // get parent process cluster and local pointer
    parent_cxy = GET_CXY( parent_xp );
    parent_ptr = (process_t *)GET_PTR( parent_xp );
    parent_pid = hal_remote_lw( XPTR( parent_cxy , &parent_ptr->pid ) );

    exec_dmsg("\n[INFO] %s : thread %x on core[%x,‰d] enters for path = %s\n",
    __FUNCTION__, CURRENT_THREAD->trdid, local_cxy, CURRENT_THREAD->core->lid, path );

    // create new process descriptor
    process = process_alloc();

    if( process == NULL )
    {
        printk("\n[ERROR] in %s : no memory / cluster = %x / ppid = %x / path = %s\n",
               __FUNCTION__ , local_cxy , parent_pid , path );
        return ENOMEM;
    }

    // get a pid from the local cluster
    error = cluster_pid_alloc( XPTR( local_cxy , process ) , &pid );

    if( error )
    {
        printk("\n[ERROR] in %s : cannot get PID / cluster = %x / ppid = %x / path = %s\n",
               __FUNCTION__ , local_cxy , parent_pid , path );
        process_free( process );
		return ENOMEM;
    }

    // initialize the process descriptor as the reference
    process_reference_init( process , pid , parent_xp );

    exec_dmsg("\n[INFO] %s : thread %x on core[%x,‰d] created process %x / path = %s\n",
    __FUNCTION__, CURRENT_THREAD->trdid, local_cxy, CURRENT_THREAD->core->lid, pid, path );

    // initialize vfs_root and vfs_cwd from parent process
    xptr_t  vfs_root_xp = hal_remote_lwd( XPTR( parent_cxy , &parent_ptr->vfs_root_xp ) );
	vfs_file_count_up( vfs_root_xp );
	process->vfs_root_xp = vfs_root_xp;

    xptr_t  vfs_cwd_xp = hal_remote_lwd( XPTR( parent_cxy , &parent_ptr->vfs_cwd_xp ) );
	vfs_file_count_up( vfs_cwd_xp );
	process->vfs_cwd_xp = vfs_cwd_xp;

    // initialize embedded fd_array from parent process
    process_fd_remote_copy( XPTR( local_cxy  , &process->fd_array ),
                            XPTR( parent_cxy , &parent_ptr->fd_array) );

    exec_dmsg("\n[INFO] %s : fd_array copied from process %x to process %x\n",
    __FUNCTION__, parent_pid , pid );

	// initialize signal manager TODO ??? [AG]
	// signal_manager_init( process );

    // register "code" and "data" vsegs as well as the process entry-point in VMM,
    // using information contained in the elf file.
	error = elf_load_process( path , process );

	if( error )
	{
		printk("\n[ERROR] in %s : failed to access elf file for process %x / path = %s\n",
		       __FUNCTION__, pid , path );
        process_destroy( process );
        return error;
	}

    exec_dmsg("\n[INFO] %s : code and data vsegs registered for process %x / path = %s\n",
    __FUNCTION__ , pid , path );

    // select a core in cluster
    lid  = cluster_select_local_core();
    core = &LOCAL_CLUSTER->core_tbl[lid];

    // initialize pthread attributes for main thread
    attr.attributes = PT_ATTR_DETACH | PT_ATTR_CLUSTER_DEFINED | PT_ATTR_CORE_DEFINED;
    attr.cxy        = local_cxy;
    attr.lid        = lid;

    // create and initialize thread descriptor
	error = thread_user_create( pid,
                                (void *)process->vmm.entry_point,
                                exec_info->args_pointers,
                                &attr,
                                &thread );
	if( error )
	{
		printk("\n[ERROR] in %s : cannot create thread for process %x / path = %s\n",
		       __FUNCTION__, pid );
        process_destroy( process );
        return error;
	}

	exec_dmsg("\n[INFO] %s : thread created for process %x on core %d in cluster %x\n",
               __FUNCTION__ , pid , core->lid , local_cxy );

    // update children list in parent process
	xlist_add_last( XPTR( parent_cxy , &parent_ptr->children_root ),
                    XPTR( local_cxy  , &process->brothers_list ) );
	hal_remote_atomic_add( XPTR( parent_cxy , &parent_ptr->children_nr) , 1 );

    // activate new thread
	thread_unblock( XPTR( local_cxy , thread ) , THREAD_BLOCKED_GLOBAL );

    exec_dmsg("\n[INFO] %s : exit for process %x\n",
                __FUNCTION__, process->pid );

	return 0;

}  // end proces_make_exec()

//////////////////////////
void process_init_create()
{
    exec_info_t   exec_info;     // structure to be passed to process_make_exec()

	error_t   error1;
	error_t   error2;
	error_t   error3;
    xptr_t    stdin_xp;
    xptr_t    stdout_xp;
    xptr_t    stderr_xp;
    uint32_t  stdin_id;
    uint32_t  stdout_id;
    uint32_t  stderr_id;

	process_dmsg("\n[INFO] %s : enters in cluster %x\n", __FUNCTION__ , local_cxy );

    // open stdin / stdout / stderr pseudo-files
	error1 = vfs_open( XPTR_NULL, CONFIG_DEV_STDIN , O_RDONLY, 0, &stdin_xp , &stdin_id  );
	error2 = vfs_open( XPTR_NULL, CONFIG_DEV_STDOUT, O_WRONLY, 0, &stdout_xp, &stdout_id );
	error3 = vfs_open( XPTR_NULL, CONFIG_DEV_STDERR, O_WRONLY, 0, &stderr_xp, &stderr_id );

	assert( ((error1 == 0) && (error2 == 0) && (error3 == 0)) , __FUNCTION__ ,
            "cannot open stdin/stdout/stderr pseudo files\n");

    assert( ((stdin_id == 0) && (stdout_id == 1) && (stderr_id == 2)) , __FUNCTION__ ,
            "bad indexes for stdin/stdout/stderr\n");

    // initialize the exec_info structure
    exec_info.parent_xp    = XPTR( local_cxy , &process_zero );
    strcpy( exec_info.path , CONFIG_PROCESS_INIT_PATH );
    exec_info.args_nr      = 0;
    exec_info.envs_nr      = 0;

    // create process_init and thread_init
	error1 = process_make_exec( &exec_info );

	assert( (error1 == 0) , __FUNCTION__ , "cannot create process_init\n");

	process_dmsg("\n[INFO] %s : exit in cluster %x\n", __FUNCTION__ , local_cxy );
		
    hal_fence();

}  // end process_init_create()

