/*
 * soclib_tty.c - soclib tty driver implementation.
 *
 * Author  Alain Greiner (2016)
 *
 * Copyright (c)  UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH..
 *
 * ALMOS-MKH. is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH. is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH.; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <dev_txt.h>
#include <chdev.h>
#include <soclib_tty.h>
#include <remote_spinlock.h>
#include <thread.h>
#include <hal_special.h>

///////////////////////////////////////
void soclib_tty_init( chdev_t * chdev )
{
    // get extended pointer on TTY-SOCLIB peripheral base address
    xptr_t tty_xp = chdev->base; 

    // get SOCLIB_TTY device cluster and local pointer
    cxy_t      tty_cxy = GET_CXY( tty_xp );
    uint32_t * tty_ptr = (uint32_t *)GET_PTR( tty_xp );
  
    // mask both TTY_RX_IRQ and TTY_TX_IRQ
    hal_remote_sw( XPTR( tty_cxy , tty_ptr + TTY_CONFIG_REG ) , 0 );

}  // soclib_tty_init()

//////////////////////////////////////////////////////////////
void __attribute__ ((noinline)) soclib_tty_cmd( xptr_t th_xp )
{
    // get client thread cluster and local pointer
    cxy_t      th_cxy = GET_CXY( th_xp );
    thread_t * th_ptr = (thread_t *)GET_PTR( th_xp );

    // get command type and extended pointer on TXT device
    uint32_t type   =         hal_remote_lw ( XPTR( th_cxy , &th_ptr->command.txt.type ) );
    xptr_t   dev_xp = (xptr_t)hal_remote_lwd( XPTR( th_cxy , &th_ptr->command.txt.dev_xp ) );
    
    // get TXT device cluster and local pointer
    cxy_t     dev_cxy = GET_CXY( dev_xp );
    chdev_t * dev_ptr = (chdev_t *)GET_PTR( dev_xp );

    // get extended pointer on SOCLIB_TTY base segment
    xptr_t tty_xp = (xptr_t)hal_remote_lwd( XPTR( dev_cxy , &dev_ptr->base ) );

    // get SOCLIB_TTY base segment cluster and local pointer
    cxy_t      tty_cxy = GET_CXY( tty_xp );
    uint32_t * tty_ptr = (uint32_t *)GET_PTR( tty_xp );

    // get TTY channel index and channel base address
    uint32_t   channel = hal_remote_lw( XPTR( dev_cxy , &dev_ptr->channel ) );
    uint32_t * base    = tty_ptr + TTY_SPAN * channel;

    if( type == TXT_READ )              // descheduling strategy for calling thead
    {
        // unmask RX_IRQ (data transfer will be done by the TTY_RX ISR)
        xptr_t config_xp = XPTR( tty_cxy , base + TTY_CONFIG_REG );
        uint32_t old = hal_remote_lw( config_xp );
        uint32_t new = old | TTY_CONFIG_RX_ENABLE;
        hal_remote_atomic_cas( config_xp , old , new );

        // Block and deschedule server thread 
        thread_block( CURRENT_THREAD , THREAD_BLOCKED_DEV_ISR );
        sched_yield();
    }
    else if( type == TXT_WRITE )        // descheduling strategy for calling thread
    {
        // unmask TX_IRQ (data transfer will be done by the TTY_TX ISR)
        xptr_t config_xp = XPTR( tty_cxy , base + TTY_CONFIG_REG );
        uint32_t old = hal_remote_lw( config_xp );
        uint32_t new = old | TTY_CONFIG_TX_ENABLE;
        hal_remote_atomic_cas( config_xp , old , new );

        // Block and deschedule server thread 
        thread_block( CURRENT_THREAD , THREAD_BLOCKED_DEV_ISR );
        sched_yield();
    }
    else if( type == TXT_SYNC_WRITE )  // busy waiting strategy for calling thread
    {
        uint32_t   status;        
        bool_t     empty;
        uint32_t   i;

        // get source buffer extended pointer & bytes count
        uint32_t count  = hal_remote_lw ( XPTR( th_cxy , &th_ptr->command.txt.count ) );
        xptr_t   buf_xp = hal_remote_lwd( XPTR( th_cxy , &th_ptr->command.txt.buf_xp ) );

        // loop on characters 
        for( i = 0 ; i < count ; i++ )
        { 
            do
            {
                // get TTY_STATUS_REG
                status = hal_remote_lw( XPTR( tty_cxy , base + TTY_STATUS_REG ) );
                empty  = ( (status & TTY_STATUS_TX_FULL) == 0 );

                if ( empty )  // TTY_TX empty => transfer one byte
                {
                    // get one byte from command buffer in client cluster
                    char byte = (char)hal_remote_lb( buf_xp + i );

                    // write byte to TTY_WRITE_REG in TTY cluster
                    hal_remote_sb( XPTR( tty_cxy , base + TTY_WRITE_REG ) , byte );
                }
            } 
            while ( empty == false );
        }
    }
}  // soclib_tty_command()


/////////////////////////////////////////////////////////////////
void __attribute__ ((noinline)) soclib_tty_isr( chdev_t * chdev )
{
    uint32_t   type;         // command type
    uint32_t   count;        // number of bytes in buffer
    xptr_t     buf_xp;       // Rextended pointer on buffer
    uint32_t   status;       // TTY terminal status
    char       byte;         // read byte
    uint32_t   i;

    // get extended pointer on client thread
    xptr_t root      = XPTR( local_cxy , &chdev->wait_root );
    xptr_t client_xp = XLIST_FIRST_ELEMENT( root , thread_t , wait_list );

    // get client thread cluster and local pointer
    cxy_t      client_cxy = GET_CXY( client_xp );
    thread_t * client_ptr = (thread_t *)GET_PTR( client_xp );

    // get command arguments
    type    = hal_remote_lw ( XPTR( client_cxy , &client_ptr->command.txt.type   ) );
    count   = hal_remote_lw ( XPTR( client_cxy , &client_ptr->command.txt.count  ) );
    buf_xp  = hal_remote_lwd( XPTR( client_cxy , &client_ptr->command.txt.buf_xp ) );

    // get SOCLIB_TTY peripheral cluster and local pointer
    cxy_t      tty_cxy = GET_CXY( chdev->base );
    uint32_t * tty_ptr = (uint32_t *)GET_PTR( chdev->base );

    // get channel base address
    uint32_t * base = tty_ptr + TTY_SPAN * chdev->channel;

    if( type == TXT_READ )              // read one single character
    {
        // get TTY_STATUS_REG
        status = hal_remote_lw( XPTR( tty_cxy , base + TTY_STATUS_REG ) );

        if( status & TTY_STATUS_RX_FULL )   // TTY_RX full => transfer one byte
        {
            // get a byte from TTY_READ_REG, and acknowledge RX_IRQ
            byte = (char)hal_remote_lb( XPTR( tty_cxy , base + TTY_READ_REG ) );

            // write it to command buffer
            hal_remote_sb( buf_xp , byte );

            // update TTY_WRITE_REG if echo mode
            if( CONFIG_TXT_ECHO_MODE )
            {
                if( (byte == '\b') || (byte == 0x7F) )
		        {
			        hal_remote_sb( XPTR( tty_cxy , base + TTY_WRITE_REG ) , '\b' );
			        hal_remote_sb( XPTR( tty_cxy , base + TTY_WRITE_REG ) , ' '  );
			        hal_remote_sb( XPTR( tty_cxy , base + TTY_WRITE_REG ) , '\b' );
		        }
                else
                {
			        hal_remote_sw( XPTR( tty_cxy , base + TTY_WRITE_REG ) , byte );
		        }
            }
        }
        else                               // buffer empty => exit ISR for retry
        {
            return;
        }
    }
    else if( type == TXT_WRITE )         // write a string
    {
        // loop on characters 
        for( i = 0 ; i < count ; i++ )
        { 
            // get TTY_STATUS_REG
            status = hal_remote_lw( XPTR( tty_cxy , base + TTY_STATUS_REG ) );

            if( (status & TTY_STATUS_TX_FULL) == 0 ) // TTY_TX empty => transfer one byte
            {
                // get one byte from command buffer
                byte = (char)hal_remote_lb( buf_xp + i );

                // write byte to TTY_WRITE_REG, and acknowledge TX_IRQ
                hal_remote_sb( XPTR( tty_cxy , base + TTY_STATUS_REG ) , byte );
            }
            else         // TTY_TX full => update command arguments and exit ISR for retry
            {
                hal_remote_sw ( XPTR( client_cxy , &client_ptr->command.txt.count ), count-i ); 
                hal_remote_swd( XPTR( client_cxy , &client_ptr->command.txt.buf_xp ), buf_xp+i );
                return;
            }
        }
    }

    // The I/O operation completed when we reach this point

    // mask both TTY_RX_IRQ and TTY_TX_IRQ
    hal_remote_sw( XPTR( tty_cxy , base + TTY_CONFIG_REG ) , 0 );

    // set I/O operation status in command
    hal_remote_sw( XPTR( client_cxy , &client_ptr->command.txt.error ) , 0 );

    // unblock server thread
    thread_unblock( XPTR( local_cxy , &chdev->server ) , THREAD_BLOCKED_DEV_ISR );

    // unblock client thread
    thread_unblock( client_xp , THREAD_BLOCKED_IO );

}  // soclib_tty_isr()


