/*
 * hal_context.c - implementation of Thread Context API for TSAR-MIPS32
 * 
 * Author  Alain Greiner    (2016)
 *
 * Copyright (c)  UPMC Sorbonne Universites
 * 
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH.is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH.is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH.; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_types.h>
#include <hal_switch.h>
#include <memcpy.h>
#include <thread.h>
#include <string.h>
#include <process.h>
#include <printk.h>
#include <vmm.h>
#include <core.h>
#include <cluster.h>
#include <hal_context.h>
#include <hal_kentry.h>

/////////////////////////////////////////////////////////////////////////////////////////
//       Define various SR values for TSAR-MIPS32 
/////////////////////////////////////////////////////////////////////////////////////////

#define SR_USR_MODE       0x0000FC13
#define SR_USR_MODE_FPU   0x2000FC13
#define SR_SYS_MODE       0x0000FC00

/////////////////////////////////////////////////////////////////////////////////////////
// This structuree defines the cpu_context for TSAR MIPS32.
// These registers are saved/restored at each context switch.
// WARNING : check the two CONFIG_CPU_CTX_SIZE & CONFIG_FPU_CTX_SIZE configuration 
//           parameterss when modifying this structure.
/////////////////////////////////////////////////////////////////////////////////////////

typedef struct hal_cpu_context_s
{
    uint32_t c0_epc;     // slot 0
    uint32_t at_01;      // slot 1
    uint32_t v0_02;      // slot 2
    uint32_t v1_03;      // slot 3
    uint32_t a0_04;      // slot 4
    uint32_t a1_05;      // slot 5
    uint32_t a2_06;      // slot 6
    uint32_t a3_07;      // slot 7

    uint32_t t0_08;      // slot 8
    uint32_t t1_09;      // slot 9
    uint32_t t2_10;      // slot 10
    uint32_t t3_11;      // slot 11
    uint32_t t4_12;      // slot 12
    uint32_t t5_13;      // slot 13
    uint32_t t6_14;      // slot 14
    uint32_t t7_15;      // slot 15

	uint32_t s0_16;      // slot 16
	uint32_t s1_17;      // slot 17
	uint32_t s2_18;      // slot 18
	uint32_t s3_19;      // slot 19
	uint32_t s4_20;      // slot 20
	uint32_t s5_21;      // slot 21
	uint32_t s6_22;      // slot 22
	uint32_t s7_23;      // slot 23

    uint32_t t8_24;      // slot 24
    uint32_t t9_25;      // slot 25
    uint32_t hi_26;      // slot 26
    uint32_t lo_27;      // slot 27
    uint32_t gp_28;      // slot 28
	uint32_t sp_29;      // slot 29
	uint32_t fp_30;      // slot 30
	uint32_t ra_31;      // slot 31

	uint32_t c2_ptpr;    // slot 32
	uint32_t c2_mode;    // slot 33

	uint32_t c0_sr;      // slot 34
	uint32_t c0_th;      // slot 35
} 
hal_cpu_context_t;

/////////////////////////////////////////////////////////////////////////////////////////
// This structure defines the fpu_context for TSAR MIPS32.
/////////////////////////////////////////////////////////////////////////////////////////

typedef struct hal_fpu_context_s
{
	uint32_t   fpu_regs[32];     
}
hal_fpu_context_t;


/////////////////////////////////////////////////////////////////////////////////////////
//        CPU context access functions
/////////////////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////////////////
// This function allocates and initializes the cpu_context stucture in thread descriptor.
// The following context slots are initialised by this function:
// GPR : a0_04 / sp_29 / fp_30 / ra_31
// CP0 : c0_sr / c0_th / c0_epc
// CP2 : c2_ptpr / c2_mode
/////////////////////////////////////////////////////////////////////////////////////////
error_t hal_cpu_context_create( thread_t * thread )
{
    kmem_req_t  req;

    assert( (sizeof(hal_cpu_context_t) <= CONFIG_CPU_CTX_SIZE) , __FUNCTION__ ,
    "inconsistent CPU context size" );

    context_dmsg("\n[DMSG] %s : enters for thread %x in process %x\n",
                 __FUNCTION__ , thread->trdid , thread->process->pid );

    // allocate memory for cpu_context
    req.type   = KMEM_CPU_CTX;
    req.flags  = AF_KERNEL | AF_ZERO;

    hal_cpu_context_t * context = (hal_cpu_context_t *)kmem_alloc( &req );
    if( context == NULL ) return ENOMEM;

    // set cpu context pointer in thread
    thread->cpu_context = (void*)context;

    // stack pointer, status register and mmu_mode depends on thread type
	uint32_t sp_29;
    uint32_t c0_sr;
    uint32_t c2_mode;
    if( thread->type == THREAD_USER )
    {
        sp_29   = ((uint32_t)thread->u_stack_base) + thread->u_stack_size;
        c0_sr   = SR_USR_MODE;
        c2_mode = 0xF;
    }
    else
    {
        sp_29   = ((uint32_t)thread->k_stack_base) + thread->k_stack_size;
        c0_sr   = SR_SYS_MODE;
        c2_mode = 0x3;
    }

    // align stack pointer on a double word boundary 
	sp_29 = (sp_29 - 8) & (~ 0x7);

    // initialise context
    context->a0_04      = (uint32_t)thread->entry_args;
	context->sp_29      = sp_29; 
	context->fp_30      = sp_29;                               // TODO check this [AG]
    context->ra_31      = (uint32_t)&hal_kentry_eret;
    context->c0_epc     = (uint32_t)thread->entry_func;
	context->c0_sr      = c0_sr;
	context->c0_th      = (uint32_t)thread; 
	context->c2_ptpr    = (uint32_t)((thread->process->vmm.gpt.ppn) >> 1);
	context->c2_mode    = c2_mode;

    context_dmsg("\n[DMSG] %s : exit for thread %x in process %x\n"
                 " - a0   = %x\n"
                 " - sp   = %x\n"
                 " - fp   = %x\n"
                 " - ra   = %x\n"
                 " - sr   = %x\n"
                 " - th   = %x\n"   
                 " - epc  = %x\n"   
                 " - ptpr = %x\n"   
                 " - mode = %x\n",  
                 __FUNCTION__ , thread->trdid , thread->process->pid,
                 context->a0_04, context->sp_29, context->fp_30, context->ra_31,
                 context->c0_sr, context->c0_th, context->c0_epc,
                 context->c2_ptpr, context->c2_mode );
    return 0;

}  // end hal_cpu_context_create()

/////////////////////////////////////////////////
void hal_cpu_context_display( thread_t * thread )
{
    hal_cpu_context_t * ctx = (hal_cpu_context_t *)thread->cpu_context;

    printk("\n***** CPU context for thread %x in process %x / cycle %d\n" 
           " gp_28   = %X    sp_29   = %X    ra_31   = %X\n" 
           " c0_sr   = %X    c0_epc  = %X    c0_th = %X\n"
           " c2_ptpr = %X    c2_mode = %X\n",
           thread->trdid, thread->process->pid, hal_time_stamp(),
           ctx->gp_28   , ctx->sp_29   , ctx->ra_31,
           ctx->c0_sr   , ctx->c0_epc  , ctx->c0_th,
           ctx->c2_ptpr , ctx->c2_mode );

}  // end hal_context_display()

/////////////////////////////////////////////////////////////////////////////////////////
// These registers are saved/restored to/from CPU context defined by <ctx> argument.
// - GPR : all, but (zero, k0, k1), plus (hi, lo)
// - CP0 : c0_th , c0_sr , C0_epc
// - CP2 : c2_ptpr , C2_mode
/////////////////////////////////////////////////////////////////////////////////////////
// old_thread  : pointer on current thread descriptor
// new_thread  : pointer on new thread descriptor
/////////////////////////////////////////////////////////////////////////////////////////
void hal_cpu_context_switch( thread_t * old_thread,
                             thread_t * new_thread )
{
    hal_cpu_context_t * ctx_old = old_thread->cpu_context;
    hal_cpu_context_t * ctx_new = new_thread->cpu_context;

    #if CONFIG_CONTEXT_DEBUG
    hal_cpu_context_display( old_thread );
    hal_cpu_context_display( new_thread );
    #endif

    // reset loadable field in new thread descriptor 
    new_thread->flags &= ~THREAD_FLAG_LOADABLE;

    hal_do_switch( ctx_old , ctx_new );
}

/////////////////////////////////////////////
error_t hal_cpu_context_copy( thread_t * dst,
                              thread_t * src )
{
    kmem_req_t  req;

    // allocate memory for dst cpu_context
    req.type   = KMEM_CPU_CTX;
    req.size   = sizeof(hal_cpu_context_t);
    req.flags  = AF_KERNEL | AF_ZERO;

    hal_cpu_context_t * dst_context = (hal_cpu_context_t *)kmem_alloc( &req );
    if( dst_context == NULL ) return ENOMEM;

    // set cpu context pointer in dst thread
    dst->cpu_context = dst_context;

    // get cpu context pointer from src thread
    hal_cpu_context_t * src_context = src->cpu_context;

    // copy CPU context from src to dst
    memcpy( dst_context , src_context , sizeof(hal_cpu_context_t) );

    return 0;

}  // end hal_cpu_context_copy()

/////////////////////////////////////////////////
void hal_cpu_context_destroy( thread_t * thread )
{
    kmem_req_t  req;

    req.type = KMEM_CPU_CTX;
    req.ptr  = thread->cpu_context;
    kmem_free( &req );

}  // end hal_cpu_context_destroy()


///////////////////////////////////////////////////
error_t hal_fpu_context_create( thread_t * thread )
{
    kmem_req_t  req;

    assert( (sizeof(hal_fpu_context_t) <= CONFIG_FPU_CTX_SIZE) , __FUNCTION__ ,
    "inconsistent FPU context size" );

    // allocate memory for uzone
    req.type   = KMEM_FPU_CTX;
    req.flags  = AF_KERNEL | AF_ZERO;

    hal_fpu_context_t * context = (hal_fpu_context_t *)kmem_alloc( &req );
    if( context == NULL ) return ENOMEM;

    // set fpu context pointer in thread
    thread->fpu_context = (void*)context;
    
    return 0;
}  // hal_fpu_context_create()

/////////////////////////////////////////////
error_t hal_fpu_context_copy( thread_t * dst,
                              thread_t * src )
{
    kmem_req_t  req;

    // allocate memory for dst fpu_context
    req.type   = KMEM_FPU_CTX;
    req.flags  = AF_KERNEL | AF_ZERO;

    hal_fpu_context_t * dst_context = (hal_fpu_context_t *)kmem_alloc( &req );
    if( dst_context == NULL ) return ENOMEM;

    // set fpu context pointer in dst thread
    dst->fpu_context = (void*)dst_context;

    // get fpu context pointer from src thread
    hal_fpu_context_t * src_context = src->fpu_context;

    // copy CPU context from src to dst
    memcpy( dst_context , src_context , sizeof(hal_fpu_context_t) );

    return 0;
}  // end hal_fpu_context_copy()

/////////////////////////////////////////////////
void hal_fpu_context_destroy( thread_t * thread )
{
    kmem_req_t  req;

    req.type = KMEM_FPU_CTX;
    req.ptr  = thread->fpu_context;
    kmem_free( &req );

}  // end hal_fpu_context_destroy()

//////////////////////////////////////////////
void hal_fpu_context_save( thread_t * thread )
{
    uint32_t ctx = (uint32_t)thread->fpu_context;

    asm volatile(
    ".set noreorder           \n"
    "swc1    $f0,    0*4(%0)  \n"   
    "swc1    $f1,    1*4(%0)  \n"   
    "swc1    $f2,    2*4(%0)  \n"   
    "swc1    $f3,    3*4(%0)  \n"   
    "swc1    $f4,    4*4(%0)  \n"   
    "swc1    $f5,    5*4(%0)  \n"   
    "swc1    $f6,    6*4(%0)  \n"   
    "swc1    $f7,    7*4(%0)  \n"   
    "swc1    $f8,    8*4(%0)  \n"   
    "swc1    $f9,    9*4(%0)  \n"   
    "swc1    $f10,  10*4(%0)  \n"   
    "swc1    $f11,  11*4(%0)  \n"   
    "swc1    $f12,  12*4(%0)  \n"   
    "swc1    $f13,  13*4(%0)  \n"   
    "swc1    $f14,  14*4(%0)  \n"   
    "swc1    $f15,  15*4(%0)  \n"   
    "swc1    $f16,  16*4(%0)  \n"   
    "swc1    $f17,  17*4(%0)  \n"   
    "swc1    $f18,  18*4(%0)  \n"   
    "swc1    $f19,  19*4(%0)  \n"   
    "swc1    $f20,  20*4(%0)  \n"   
    "swc1    $f21,  21*4(%0)  \n"   
    "swc1    $f22,  22*4(%0)  \n"   
    "swc1    $f23,  23*4(%0)  \n"   
    "swc1    $f24,  24*4(%0)  \n"   
    "swc1    $f25,  25*4(%0)  \n"   
    "swc1    $f26,  26*4(%0)  \n"   
    "swc1    $f27,  27*4(%0)  \n"   
    "swc1    $f28,  28*4(%0)  \n"   
    "swc1    $f29,  29*4(%0)  \n"   
    "swc1    $f30,  30*4(%0)  \n"   
    "swc1    $f31,  31*4(%0)  \n"   
    ".set reorder             \n"
    : : "r"(ctx) );

}  // end hal_cpu_context_save()

/////////////////////////////////////////////////
void hal_fpu_context_restore( thread_t * thread )
{
    uint32_t ctx = (uint32_t)thread->fpu_context;

    asm volatile(
    ".set noreorder           \n"
    "lwc1    $f0,    0*4(%0)  \n"   
    "lwc1    $f1,    1*4(%0)  \n"   
    "lwc1    $f2,    2*4(%0)  \n"   
    "lwc1    $f3,    3*4(%0)  \n"   
    "lwc1    $f4,    4*4(%0)  \n"   
    "lwc1    $f5,    5*4(%0)  \n"   
    "lwc1    $f6,    6*4(%0)  \n"   
    "lwc1    $f7,    7*4(%0)  \n"   
    "lwc1    $f8,    8*4(%0)  \n"   
    "lwc1    $f9,    9*4(%0)  \n"   
    "lwc1    $f10,  10*4(%0)  \n"   
    "lwc1    $f11,  11*4(%0)  \n"   
    "lwc1    $f12,  12*4(%0)  \n"   
    "lwc1    $f13,  13*4(%0)  \n"   
    "lwc1    $f14,  14*4(%0)  \n"   
    "lwc1    $f15,  15*4(%0)  \n"   
    "lwc1    $f16,  16*4(%0)  \n"   
    "lwc1    $f17,  17*4(%0)  \n"   
    "lwc1    $f18,  18*4(%0)  \n"   
    "lwc1    $f19,  19*4(%0)  \n"   
    "lwc1    $f20,  20*4(%0)  \n"   
    "lwc1    $f21,  21*4(%0)  \n"   
    "lwc1    $f22,  22*4(%0)  \n"   
    "lwc1    $f23,  23*4(%0)  \n"   
    "lwc1    $f24,  24*4(%0)  \n"   
    "lwc1    $f25,  25*4(%0)  \n"   
    "lwc1    $f26,  26*4(%0)  \n"   
    "lwc1    $f27,  27*4(%0)  \n"   
    "lwc1    $f28,  28*4(%0)  \n"   
    "lwc1    $f29,  29*4(%0)  \n"   
    "lwc1    $f30,  30*4(%0)  \n"   
    "lwc1    $f31,  31*4(%0)  \n"   
    ".set reorder             \n"
    : : "r"(ctx) );

} // end hal_cpu_context_restore()

/////////////////////////////////////
void hal_fpu_context_dup( xptr_t dst,
                          xptr_t src )
{
	hal_remote_memcpy( dst , src , sizeof(hal_fpu_context_t) );
}

