/*
 * hal_exception.c - implementation of exception handler for TSAR-MIPS32.
 * 
 * Author   Alain Greiner (2016, 2017)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_types.h>
#include <hal_irqmask.h>
#include <hal_special.h>
#include <hal_exception.h>
#include <thread.h>
#include <printk.h>
#include <chdev.h>
#include <vmm.h>
#include <errno.h>
#include <scheduler.h>
#include <core.h>
#include <signal.h>
#include <syscalls.h>
#include <remote_spinlock.h>
#include <hal_kentry.h>


//////////////////////////////////////////////////////////////////////////////////////////
//  Extern global variables
//////////////////////////////////////////////////////////////////////////////////////////

extern   chdev_directory_t    chdev_dir;  // allocated in the kernel_init.c file.

//////////////////////////////////////////////////////////////////////////////////////////
// This enum defines the global exception types after analysis by the exception handler.
//////////////////////////////////////////////////////////////////////////////////////////

typedef enum
{
    EXCP_NON_FATAL,
    EXCP_USER_ERROR,
    EXCP_KERNEL_PANIC,
}
exception_handling_type_t;

//////////////////////////////////////////////////////////////////////////////////////////
// This enum defines the mask values for an MMU exception code reported by the mips32.
//////////////////////////////////////////////////////////////////////////////////////////

typedef enum
{
    MMU_WRITE_PT1_UNMAPPED        = 0x0001,
    MMU_WRITE_PT2_UNMAPPED        = 0x0002,
    MMU_WRITE_PRIVILEGE_VIOLATION = 0x0004,
    MMU_WRITE_ACCESS_VIOLATION    = 0x0008,
    MMU_WRITE_UNDEFINED_XTN       = 0x0020,
    MMU_WRITE_PT1_ILLEGAL_ACCESS  = 0x0040,
    MMU_WRITE_PT2_ILLEGAL_ACCESS  = 0x0080,
    MMU_WRITE_DATA_ILLEGAL_ACCESS = 0x0100,

    MMU_READ_PT1_UNMAPPED         = 0x1001,
    MMU_READ_PT2_UNMAPPED         = 0x1002,
    MMU_READ_PRIVILEGE_VIOLATION  = 0x1004,
    MMU_READ_EXEC_VIOLATION       = 0x1010,
    MMU_READ_UNDEFINED_XTN        = 0x1020,
    MMU_READ_PT1_ILLEGAL_ACCESS   = 0x1040,
    MMU_READ_PT2_ILLEGAL_ACCESS   = 0x1080,
    MMU_READ_DATA_ILLEGAL_ACCESS  = 0x1100,
}
mmu_exception_subtype_t;

//////////////////////////////////////////////////////////////////////////////////////////
// This enum defines the relevant values for XCODE field in mips32 CP0_CR register.
//////////////////////////////////////////////////////////////////////////////////////////

typedef enum
{
    XCODE_ADEL = 0x4,        // Illegal address for data load 
    XCODE_ADES = 0x5,        // Illegal address for data store
    XCODE_IBE  = 0x6,        // Instruction MMU exception       (can be NON-FATAL)
    XCODE_DBE  = 0x7,        // Data MMU exception              (can be NON-FATAL)
    XCODE_RI   = 0xA,        // Reserved instruction exception
    XCODE_CPU  = 0xB,        // Coprocessor unusable exception  (can be NON-FATAl)
    XCODE_OVR  = 0xC,        // Arithmetic Overflow exception
}
xcode_values_t;

//////////////////////////////////////////////////////////////////////////////////////////
// This function is called when a FPU Coprocessor Unavailable exception has been 
// detected for the calling thread. 
// It enables the FPU, It saves the current FPU context in the current owner thread 
// descriptor if required, and restore the FPU context from the calling thread descriptor.
//////////////////////////////////////////////////////////////////////////////////////////
// @ this     : pointer on faulty thread descriptor.
// @ return always EXCP_NON_FATAL
//////////////////////////////////////////////////////////////////////////////////////////
error_t hal_fpu_exception( thread_t * this )
{
	core_t   * core = this->core; 

    // enable FPU  
	hal_fpu_enable();

    // save FPU context in current owner thread if required 
	if( core->fpu_owner != NULL )
    {
        if( core->fpu_owner != this )
	    {
		    hal_fpu_context_save ( core->fpu_owner->fpu_context );
        }
	}

    // attach the FPU to the requesting thread
	hal_fpu_context_restore( this->fpu_context );
	core->fpu_owner = this;

	return EXCP_NON_FATAL;

}  // end hal_fpu_exception()

//////////////////////////////////////////////////////////////////////////////////////////
// This function is called when an MMU exception has been detected. 
// It get the relevant exception arguments from the MMU.
// It signal a fatal error in case of illegal access. In case of page unmapped
// it checks that the faulty address belongs to a registered vseg. It update the local
// vseg list from the reference cluster if required, and signal a fatal user error
// in case of illegal virtual address. Finally, it updates the local page table from the
// reference cluster.
//////////////////////////////////////////////////////////////////////////////////////////
// @ this     : pointer on faulty thread descriptor.
// @ is_ins   : IBE if true / DBE if false.
// @ return EXCP_NON_FATAL / EXCP_USER_ERROR / EXCP_KERNEL_PANIC
//////////////////////////////////////////////////////////////////////////////////////////
error_t hal_mmu_exception( thread_t * this,
                           bool_t     is_ins ) 
{
	process_t      * process;
    error_t          error;

    uint32_t         mmu_ins_excp_code;
    uint32_t         mmu_ins_bad_vaddr;
    uint32_t         mmu_dat_excp_code;
    uint32_t         mmu_dat_bad_vaddr;

    uint32_t         bad_vaddr;
    uint32_t         excp_code;
	
    process = this->process;

    // get relevant values from MMU
	hal_get_mmu_excp( &mmu_ins_excp_code,
	                  &mmu_ins_bad_vaddr,
	                  &mmu_dat_excp_code, 
	                  &mmu_dat_bad_vaddr );

    // get exception code and faulty vaddr, depending on IBE/DBE
    if( is_ins )
    {
        excp_code = mmu_ins_excp_code;
        bad_vaddr = mmu_ins_bad_vaddr;
    }
    else 
    {
        excp_code = mmu_dat_excp_code;
        bad_vaddr = mmu_dat_bad_vaddr;
    }

 excp_dmsg("\n[DBG] %s : core[%x,%d] / is_ins %d / code %x / vaddr %x\n",
__FUNCTION__ , local_cxy , this->core->lid , is_ins, excp_code, bad_vaddr );

   // analyse exception code
    switch( excp_code )
    {
        case MMU_WRITE_PT1_UNMAPPED:      // non fatal
        case MMU_WRITE_PT2_UNMAPPED:
        case MMU_READ_PT1_UNMAPPED:
        case MMU_READ_PT2_UNMAPPED:
        {
            // try to map the unmapped PTE
            error = vmm_handle_page_fault( process, 
                                           bad_vaddr >> CONFIG_PPM_PAGE_SHIFT );  // vpn
            if( error )   
            {
                printk("\n[ERROR] in %s for thread %x : cannot map vaddr = %x\n",
                __FUNCTION__ , this->trdid , bad_vaddr );

		        return EXCP_USER_ERROR;
            } 
            else            // page fault successfull
            {

excp_dmsg("\n[DBG] %s : core[%x,%d] / page-fault handled for vaddr = %x\n",
__FUNCTION__ , local_cxy , this->core->lid , bad_vaddr );
  
                return EXCP_NON_FATAL;
            }
        }
        case MMU_WRITE_PRIVILEGE_VIOLATION:  // illegal access user error
        case MMU_READ_PRIVILEGE_VIOLATION:
        {
            printk("\n[ERROR] in %s for thread %x : illegal user access to vaddr = %x\n",
            __FUNCTION__ , this->trdid , bad_vaddr );

            return EXCP_USER_ERROR;
        }
        case MMU_WRITE_ACCESS_VIOLATION:     // user error or Copy-on-Write
        {
            // access page table to get GPT_COW flag
            bool_t cow = hal_gpt_pte_is_cow( &(process->vmm.gpt),
                                             bad_vaddr >> CONFIG_PPM_PAGE_SHIFT );  // vpn

            if( cow )                        // Copy-on-Write 
            {
                // try to allocate and copy the page
                error = vmm_copy_on_write( process,
                                           bad_vaddr >> CONFIG_PPM_PAGE_SHIFT );  // vpn
                if( error )
                {
                    printk("\n[ERROR] in %s for thread %x : cannot cow vaddr = %x\n",
                    __FUNCTION__ , this->trdid , bad_vaddr );

		            return EXCP_USER_ERROR;
                }
                else         // Copy on write successfull
                {

excp_dmsg("\n[DBG] %s : core[%x,%d] / copy-on-write handled for vaddr = %x\n",
__FUNCTION__ , local_cxy , this->core->lid , bad_vaddr );

                    return EXCP_NON_FATAL;
                } 
            }
            else                             // non writable user error
            {
                printk("\n[ERROR] in %s for thread %x : write to non-writable vaddr = %x\n",
                __FUNCTION__ , this->trdid , bad_vaddr );

                return EXCP_USER_ERROR;
            }
        }
        case MMU_READ_EXEC_VIOLATION:        // user error
        {
            printk("\n[ERROR] in %s for thread %x : read to non-executable vaddr = %x\n",
            __FUNCTION__ , this->trdid , bad_vaddr );

            return EXCP_USER_ERROR;
        }
        default:                             // this is a kernel error => panic    
        {
            printk("\n[PANIC] in %s for thread %x : kernel exception = %x / vaddr = %x\n",
            __FUNCTION__ , this->trdid , excp_code , bad_vaddr );

            return EXCP_KERNEL_PANIC;
        }
    }  
} // end hal_mmu_exception()

//////////////////////////////////////////////////////////////////////////////////////////
// This static function prints on the kernel terminal the saved context (core registers)
// and the thread state of a faulty thread.
//////////////////////////////////////////////////////////////////////////////////////////
// @ this     : pointer on faulty thread descriptor.
// @ regs_tbl : pointer on register array.
// @ error    : EXCP_USER_ERROR or EXCP_KERNEL_PANIC
//////////////////////////////////////////////////////////////////////////////////////////
static void hal_exception_dump( thread_t * this,
                                reg_t    * regs_tbl,
                                error_t    error )
{
    uint32_t  save_sr;
    core_t  * core = this->core;

    // get pointers on TXT0 chdev
    xptr_t    txt0_xp  = chdev_dir.txt_tx[0];
    cxy_t     txt0_cxy = GET_CXY( txt0_xp );
    chdev_t * txt0_ptr = GET_PTR( txt0_xp );

    // get extended pointer on remote TXT0 chdev lock
    xptr_t  lock_xp = XPTR( txt0_cxy , &txt0_ptr->wait_lock );

    // get TXT0 lock in busy waiting mode
    remote_spinlock_lock_busy( lock_xp , &save_sr );

    if( error == EXCP_USER_ERROR )
    {
        nolock_printk("\n========= USER ERROR / core[%x,%d] / cycle %d ==============\n",
        local_cxy , core->lid , (uint32_t)hal_get_cycles() );
    }
    else
    {
        nolock_printk("\n======= KERNEL PANIC / core[%x,%d] / cycle %d ==============\n",
        local_cxy , core->lid , (uint32_t)hal_get_cycles() );
    }

	nolock_printk("  thread type = %s / trdid = %x / pid %x / core[%x,%d]\n"
           "  local locks = %d / remote locks = %d / blocked_vector = %X\n\n",
           thread_type_str(this->type), this->trdid, this->process->pid, local_cxy,
           this->core->lid, this->local_locks, this->remote_locks, this->blocked );

	nolock_printk("cp0_cr %X   cp0_epc %X   cp0_sr %X   cp2_mode %X\n",
		   regs_tbl[UZ_CR], regs_tbl[UZ_EPC], regs_tbl[UZ_SR], regs_tbl[UZ_MODE]);

    nolock_printk("at_01 %X  v0_2  %X  v1_3  %X  a0_4   %X  a1_5   %X\n",
   	       regs_tbl[UZ_AT],regs_tbl[UZ_V0],regs_tbl[UZ_V1],regs_tbl[UZ_A0],regs_tbl[UZ_A1]);

    nolock_printk("a2_6  %X  a3_7  %X  t0_8  %X  t1_9   %X  t2_10  %X\n",
		   regs_tbl[UZ_A2],regs_tbl[UZ_A3],regs_tbl[UZ_T0],regs_tbl[UZ_T1],regs_tbl[UZ_T2]);
  
    nolock_printk("t3_11 %X  t4_12 %X  t5_13 %X  t6_14  %X  t7_15  %X\n",
		   regs_tbl[UZ_T3],regs_tbl[UZ_T4],regs_tbl[UZ_T5],regs_tbl[UZ_T6],regs_tbl[UZ_T7]);

    nolock_printk("s0_16 %X  s1_17 %X  s2_18 %X  s3_19  %X  s4_20  %X\n",
		   regs_tbl[UZ_S0],regs_tbl[UZ_S1],regs_tbl[UZ_S2],regs_tbl[UZ_S3],regs_tbl[UZ_S4]);
  
    nolock_printk("s5_21 %X  s6_22 %X  s7_23 %X  s8_24  %X  ra_25  %X\n",
	       regs_tbl[UZ_S5],regs_tbl[UZ_S6],regs_tbl[UZ_S7],regs_tbl[UZ_T8],regs_tbl[UZ_T9]);

    nolock_printk("gp_28 %X  sp_29 %X  S8_30 %X  ra_31  %X\n",
		   regs_tbl[UZ_GP],regs_tbl[UZ_SP],regs_tbl[UZ_S8],regs_tbl[UZ_RA]);

    // release the lock
    remote_spinlock_unlock_busy( lock_xp , save_sr );

}  // end hal_exception_dump()


///////////////////////////////////////////////////////////////////////////////
// TODO replace the hal_core_sleep() by the generic panic() function.
///////////////////////////////////////////////////////////////////////////////
void hal_do_exception( thread_t * this, 
                       reg_t    * regs_tbl )
{
	error_t             error;
	uint32_t            excCode;                  // 4 bits XCODE from CP0_CR

    // get 4 bits XCODE from CP0_CR register
	excCode        = (regs_tbl[UZ_CR] >> 2) & 0xF;

excp_dmsg("\n[DBG] %s : core[%x,%d] / thread %x in process %x / xcode %x / cycle %d\n",
__FUNCTION__, local_cxy, this->core->lid, this->trdid, 
this->process->pid, excCode, (uint32_t)hal_get_cycle() );

	switch(excCode)
	{
        case XCODE_DBE:     // can be non fatal
        {
		    error = hal_mmu_exception( this , false );  // data MMU exception
            break;
        }
	    case XCODE_IBE:     // can be non fatal
        {
		    error = hal_mmu_exception( this , true );   // ins MMU exception
		    break;
        }
	    case XCODE_CPU:    // can be non fatal
        {
            if( ((regs_tbl[UZ_CR] >> 28) & 0x3) == 1 )  // unavailable FPU
            {
                error = hal_fpu_exception( this );
            }
            else
            {
		        error = EXCP_USER_ERROR;
            }
		    break;
        }
        case XCODE_OVR:    // user fatal error
        case XCODE_RI:     // user fatal error
        case XCODE_ADEL:   // user fatal error
        case XCODE_ADES:   // user fatal error
        {
		    error = EXCP_USER_ERROR;
	        break;
        }
        default:
        {
		    error = EXCP_KERNEL_PANIC;
        }
	}
    
    // analyse error code
	if( error == EXCP_USER_ERROR )          //  user error => kill user process
	{
        hal_exception_dump( this , regs_tbl , error );

        // FIXME : replace this loop by sys_kill()
        while( 1 ) asm volatile ("nop");
        // sys_kill( this->process->pid , SIGKILL );
	}
    else if( error == EXCP_KERNEL_PANIC )   // kernel error => kernel panic
    {
        hal_exception_dump( this , regs_tbl , error );
        hal_core_sleep();
    }

excp_dmsg("\n[DBG] %s : core[%x,%d] exit / thread %x in process %x / cycle %d\n",
__FUNCTION__, local_cxy, this->core->lid, this->trdid, this->process->pid, hal_time_stamp() );

}  // end hal_do_exception()


