/*
 * sys_write.c - write bytes to a file
 * 
 * Author        Alain Greiner (2016,2017)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <kernel_config.h>
#include <hal_types.h>
#include <hal_uspace.h>
#include <hal_special.h>
#include <errno.h>
#include <vfs.h>
#include <thread.h>
#include <printk.h>
#include <process.h>

/* TODO: concurrent user page(s) unmap need to be handled [AG] */

//////////////////////////////////
int sys_write( uint32_t   file_id,
               void     * vaddr,
               uint32_t   count )
{
    error_t      error;
    paddr_t      paddr;                // unused, but required for user space checking
	xptr_t       file_xp;              // remote file extended pointer
    uint32_t     nbytes;               // number of bytes actually written

	uint32_t     tm_start;
	uint32_t     tm_end;

	tm_start = hal_get_cycles();

	thread_t   * this = CURRENT_THREAD;
	process_t  * process = this->process;
 
    // check file_id argument
	if( file_id >= CONFIG_PROCESS_FILE_MAX_NR )
	{
        printk("\n[ERROR] in %s : illegal file descriptor index\n", __FUNCTION__ );
        this->errno = EBADFD;
		return -1;
	}

    // check user buffer in user space
    error = vmm_v2p_translate( false , vaddr , &paddr );

    if ( error )
    {
        printk("\n[ERROR] in %s : user buffer unmapped = %x\n",
        __FUNCTION__ , (intptr_t)vaddr );
		this->errno = EINVAL;
		return -1;
    }

    // get extended pointer on remote file descriptor
    file_xp = process_fd_get_xptr( process , file_id );

    if( file_xp == XPTR_NULL )
    {
        printk("\n[ERROR] in %s : undefined file descriptor index = %d in process %x\n",
        __FUNCTION__ , file_id , process->pid );
		this->errno = EBADFD;
		return -1;
    }

    // get file descriptor cluster and local pointer
    vfs_file_t * file_ptr = (vfs_file_t *)GET_PTR( file_xp );
    cxy_t        file_cxy = GET_CXY( file_xp );

    // check file writable
    uint32_t attr = hal_remote_lw( XPTR( file_cxy , &file_ptr->attr ) );
    if( (attr & FD_ATTR_WRITE_ENABLE) == 0 )
	{
        printk("\n[ERROR] in %s : file %d not writable in process %x\n",
        __FUNCTION__ , file_id , process->pid );
		this->errno = EBADFD;
		return -1;
	}
   
    // get file type
    vfs_inode_type_t type = hal_remote_lw( XPTR( file_cxy , &file_ptr->type ) );

    // action depend on file type
    if( type == INODE_TYPE_FILE )      // transfer count bytes to file mapper 
    {
        nbytes = vfs_user_move( false,               // from buffer to mapper
                                file_xp,
                                vaddr, 
                                count );
    }
    else if( type == INODE_TYPE_DEV )  // transfer count bytes to device
    {
        nbytes = devfs_user_move( false,             // from buffer to device
                                 file_xp,
                                 vaddr,
                                 count );
    }
    else
    {
        nbytes = 0;
        panic("file type %d non supported", type );
    }

    if( nbytes != count )
    {
        printk("\n[ERROR] in %s cannot write data to file %d in process %x\n",
        __FUNCTION__ , file_id , process->pid );
        this->errno = error;
        return -1;
    }

    hal_fence();

    tm_end = hal_get_cycles();

syscall_dmsg("\n[DBG] %s : core[%x,%d] / thread %x / nbytes = %d / cycle %d\n"
" first byte = %c / file_id = %d / cost = %d\n",
__FUNCTION__ , local_cxy , this->core->lid , this->trdid , nbytes , tm_start ,
*((char *)(intptr_t)paddr) , file_id , tm_end - tm_start );
 
	return nbytes;

}  // end sys_write()
