/*
 * sys_thread_join.c - passive wait on the end of a given thread.
 * 
 * Authors    Alain Greiner (2016,2017)
 *
 * Copyright (c) 2011,2012 UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_types.h>
#include <hal_remote.h>
#include <hal_special.h>
#include <thread.h>
#include <vmm.h>
#include <scheduler.h>
#include <errno.h>
#include <printk.h>
#include <remote_spinlock.h>

///////////////////////////////////////
int sys_thread_join ( trdid_t    trdid,
                      void    ** exit_value )
{
    xptr_t        target_xp;
    thread_t    * target_ptr;
    cxy_t         target_cxy;
    ltid_t        target_ltid;
    uint32_t      flags;        // target thread flags
    intptr_t      value;        // value returned by target thread
    paddr_t       paddr;        // required for vmm_v2p_translate()

	thread_t  * this    = CURRENT_THREAD;
    process_t * process = this->process;

    // get target thread ltid and cxy
    target_ltid = LTID_FROM_TRDID( trdid );
    target_cxy  = CXY_FROM_TRDID( trdid );

    // check trdid argument
	if( (target_ltid >= CONFIG_THREAD_MAX_PER_CLUSTER) || cluster_is_undefined( target_cxy ) )  
	{
        printk("\n[ERROR] in %s : illegal trdid argument\n", __FUNCTION__ );
		this->errno = EINVAL;
		return -1;
	}

    // check exit_value argument
	if( (exit_value != NULL) && (vmm_v2p_translate( false , exit_value , &paddr ) != 0 ) )
	{
        printk("\n[ERROR] in %s : illegal exit_value argument\n", __FUNCTION__ );
		this->errno = EINVAL;
		return -1;
	}

    // check target thread != this thread
    if( this->trdid == trdid )
    {
        printk("\n[ERROR] in %s : this thread == target thread\n", __FUNCTION__ );
        this->errno = EDEADLK;
        return -1;
    }

    // get extended pointer on target thread
	target_xp  = thread_get_xptr( process->pid , trdid );

    if( target_xp == XPTR_NULL )
    {
        printk("\n[ERROR] in %s : target thread not found\n", __FUNCTION__ );
        this->errno = ESRCH;
        return -1;
    }

    // get cluster and local pointer on target thread
    target_ptr = (thread_t *)GET_PTR( target_xp );

    // check target thread joinable
    flags = hal_remote_lw( XPTR( target_cxy , &target_ptr->flags ) );
    if( flags & THREAD_FLAG_DETACHED )
    {
        printk("\n[ERROR] in %s : target thread not joinable\n", __FUNCTION__ );
        this->errno = EINVAL; 
        return -1;
    }

    // check kernel stack overflow
    if( target_ptr->signature != THREAD_SIGNATURE )
    {
        printk("\n[PANIC] in %s : kernel stack overflow\n", __FUNCTION__ );
        hal_core_sleep();
    }

    // wait target thread exit
    while( 1 )
    {
        // take the target thread lock protecting flags      
        remote_spinlock_lock( XPTR( target_cxy , &target_ptr->flags_lock ) );

        // get the remote thread flags
        flags = hal_remote_lw( XPTR( target_cxy , &target_ptr->flags ) );

        // check the EXIT flag
        if( flags & THREAD_FLAG_EXIT )   // target made an exit
        {
            // unblock the target thread
            thread_unblock( target_xp , THREAD_BLOCKED_EXIT );

            // release the target thread lock protecting flags      
            remote_spinlock_unlock( XPTR( target_cxy , &target_ptr->flags_lock ) );

            // exit while
            break;
        }
        else                             // no exit done by target thread
        {
            // set the JOIN flag in target thread
            hal_remote_atomic_or( XPTR( target_xp , &target_ptr->flags ) , 
                                  THREAD_BLOCKED_JOIN );

            // block this thread 
            thread_block( this , THREAD_BLOCKED_JOIN );

            // release the target thread lock protecting flags
	        remote_spinlock_unlock( XPTR( target_cxy , &target_ptr->flags_lock ) );

            // deschedule
            sched_yield();
        }
    }

    // return exit_value from target thread descriptor
    value = (intptr_t)hal_remote_lpt( XPTR( target_cxy , &target_ptr->exit_value ) );
    *exit_value = (void *)value;
    return 0;

}  // end sys_thread_join()
