/*
 * hal_kentry.S - Interrupt / Exception / Syscall kernel entry point for MIPS32
 * 
 * AUthors   Ghassan Almaless (2007,2008,2009,2010,2011,2012)
 *           Mohamed Lamine Karaoui (2015)
 *           Alain Greiner (2017)
 *
 * Copyright (c) UPMC Sorbonne Universites
 * 
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#define      UZ_MODE         0                 
#define      UZ_AT           1
#define      UZ_V0           2
#define      UZ_V1           3
#define      UZ_A0           4 
#define      UZ_A1           5
#define      UZ_A2           6
#define      UZ_A3           7
#define      UZ_T0           8
#define      UZ_T1           9
#define      UZ_T2           10
#define      UZ_T3           11
#define      UZ_T4           12
#define      UZ_T5           13
#define      UZ_T6           14
#define      UZ_T7           15
#define      UZ_T8           16
#define      UZ_T9           17
#define      UZ_S0           18
#define      UZ_S1           19
#define      UZ_S2           20
#define      UZ_S3           21
#define      UZ_S4           22
#define      UZ_S5           23
#define      UZ_S6           24
#define      UZ_S7           25
#define      UZ_S8           26
#define      UZ_GP           27
#define      UZ_RA           28
#define      UZ_EPC          29
#define      UZ_CR           30
#define      UZ_SP           31
#define      UZ_SR           32
#define      UZ_LO           33
#define      UZ_HI           34

#define      UZ_REGS         35

#include <kernel_config.h>

	.section   .kentry, "ax", @progbits

	.extern    hal_do_interrupt
	.extern    hal_do_exception
	.extern    hal_do_syscall
	.extern    cluster_core_kernel_enter
	.extern    cluster_core_kentry_exit

	.org       0x180

	.global    hal_kentry_enter
    .global    hal_kentry_eret

	.set       noat
	.set       noreorder

#---------------------------------------------------------------------------------
# Kernel Entry point for Interrupt / Exception / Syscall
# The c2_dext and c2_iext CP2 registers must have been previously set
# to "local_cxy", because the kernel run with MMU desactivated.
#---------------------------------------------------------------------------------

hal_kentry_enter:

	mfc0    $26,    $12                 # get c0_sr 
	andi    $26,    $26,  0x10          # test User Mode bit
	beq     $26,    $0,	  kernel_mode   # jump if core already in kernel
	ori     $27,    $0,	  0x3           # $27 <= code for MMU OFF
	
#---------------------------------------------------------------------------------------	
# This code is executed when the core is in user mode:
# - save current c2_mode in $26.
# - set MMU OFF.
# - save user stack pointer in $27.
# - set kernel stack pointer in $29.

user_mode:

    mfc2    $26,    $1                  # $26 <= c2_mode
	mtc2    $27,    $1			        # set MMU OFF
    move    $27,    $29                 # $27 <= user stack pointer
	mfc0    $29,    $4,   2             # get pointer on thread descriptor from c0_th
    addi    $29,    $29,    CONFIG_THREAD_DESC_SIZE
    addi    $29,    $29,    -8          # $29 <= kernel stack pointer
    j       unified_mode
    nop

#---------------------------------------------------------------------------------------	
# This code is executed when the core is already in kernel mode:
# - save current c2_mode in $26.
# - set MMU OFF.
# - save current kernel stack pointer in $27.

kernel_mode:

    mfc2    $26,    $1                  # $26 <= c2_mode
	mtc2    $27,    $1			        # set MMU OFF
    move    $27,    $29                 # $27 <= current kernel stack pointer

#---------------------------------------------------------------------------------------	
# This code is executed in both modes (user or kernel):
# The assumptions are:
# - c2_mode contains the MMU OFF value.
# - $26 contains the previous c2_mode value.
# - $27 contains the previous sp value (can be usp or ksp).
# - $29 contains the kernel stack pointer.
# We execute the following actions:
# - allocate an uzone in kernel stack, incrementing $29
# - save relevant registers to uzone.
# - set the SR in kernel mode: IRQ disabled, clear exl.
# - signal the kernel entry.

unified_mode:

	addiu   $29,    $29,  -(UZ_REGS*4)	# allocate uzone in kernel stack 

	sw      $1,	    (UZ_AT*4)($29)
	sw      $2,     (UZ_V0*4)($29)
	sw      $3,     (UZ_V1*4)($29)
	sw      $4,     (UZ_A0*4)($29)
	sw      $5,     (UZ_A1*4)($29)
	sw      $6,     (UZ_A2*4)($29)
	sw      $7,     (UZ_A3*4)($29)
	sw      $8,     (UZ_T0*4)($29)
	sw      $9,     (UZ_T1*4)($29)
	sw      $10,	(UZ_T2*4)($29)
	sw      $11,	(UZ_T3*4)($29)
	sw      $12,	(UZ_T4*4)($29)
	sw      $13,	(UZ_T5*4)($29)
	sw      $14,	(UZ_T6*4)($29)
	sw      $15,	(UZ_T7*4)($29)
	sw      $16,	(UZ_S0*4)($29)
	sw      $17,	(UZ_S1*4)($29)
	sw	    $18,	(UZ_S2*4)($29)
	sw	    $19,	(UZ_S3*4)($29)
	sw	    $20,	(UZ_S4*4)($29)
	sw	    $21,	(UZ_S5*4)($29)
	sw	    $22,	(UZ_S6*4)($29)
	sw	    $23,	(UZ_S7*4)($29)
	sw      $24,	(UZ_T8*4)($29)
	sw      $25,	(UZ_T9*4)($29)

	sw      $26,    (UZ_MODE*4)($29)    # save c2_mode
	sw      $27,	(UZ_SP*4)($29)	    # save sp

	sw	    $28,	(UZ_GP*4)($29)
	sw	    $30,	(UZ_S8*4)($29)
	sw	    $31,	(UZ_RA*4)($29)

	mfc0	$16,    $14
	sw      $16,	(UZ_EPC*4)($29)	    # save c0_epc
	mflo	$14
	sw      $14,	(UZ_LO*4)($29)      # save lo
	mfhi	$15 
	sw      $15,	(UZ_HI*4)($29)		# save hi
	mfc0	$18,	$12 
	sw	    $18,	(UZ_SR*4)($29)		# save c0_sr
	mfc0    $17,    $13
	sw      $17,	(UZ_CR*4)($29)		# save c0_cr
	mfc2    $26,    $1

	srl	    $3,     $18,  5
	sll     $3,	    $3,   5	
	mtc0	$3,	    $12			        # set new sr

#---------------------------------------------------------------------------------------
# This code call the relevant Interrupt / Exception / Syscall handler,
# depending on XCODE in CP0_CR.
# assumption: $29 contains the kernel stack pointer, that is the uzone base.
# The three handlers take the same two arguments: thread pointer and uzone pointer.
# The uzone pointer is saved in $19 to be used by kentry_exit.

	mfc0    $17,    $13                 # $17 <= CR
	andi    $17,    $17,   0x3F         # $17 <= XCODE

	mfc0    $4,     $4,   2             # $4 <= thread pointer (first arg)
	or	    $5,	    $0,   $29		    # $5 <= uzone pointer (second arg)
	or	    $19,    $0,	  $29	        # $19 <= &uzone (for kentry_exit)

	ori	    $8,     $0,   0x20          # $8 <= cause syscall
    beq	    $8,	    $17,  cause_sys     # go to syscall handler
    nop
	beq     $17,   	$0,	  cause_int     # go to interrupt handler
    nop

cause_excp:
	la      $1,	    hal_do_exception
	jalr	$1	                        # call exception handler
	addiu	$29,	$29,  -8	        # hal_do_exception has 2 args
	addiu	$29,	$29,  8
	j       kentry_exit                 # jump to kentry_exit
    nop

cause_sys:
	la	    $1, 	hal_do_syscall
	jalr	$1                          # call syscall handler                 
	addiu	$29,	$29,  -8            # hal_do_syscall has 2 args
	addiu	$29,	$29,  8
	j	    kentry_exit                 # jump to kentry_exit
	nop 
	
cause_int:
	la	    $1,	    hal_do_interrupt
	jalr    $1                          # call interrupt handler
	addiu	$29,	$29,  -8            # hal_do_interrupt has 2 args
	addiu	$29,	$29,  8

# -----------------------------------------------------------------------------------
# Kernel exit
# The pointer on uzone is supposed to be stored in $19
# -----------------------------------------------------------------------------------
kentry_exit:

	# restore registers from uzone
	or	    $27,    $0,	$19             # $27 <= &uzone

	lw	    $29,	(UZ_SP*4)($27)		# restore SP from uzone
	lw	    $16,	(UZ_EPC*4)($27)	       
	mtc0	$16,	$14			        # restore EPC from uzone
	lw	    $16,    (UZ_HI*4)($27)
	mthi	$16				            # restore HI from uzone
	lw	    $16,    (UZ_LO*4)($27)
	mtlo	$16				            # restore LO from uzone

	lw	    $17,	(UZ_SR*4)($27)		# get saved SR value from uzone
	andi	$17,	$17,	0x1F        # keep only the 5 LSB bits
	mfc0	$26,	$12			        # get current SR value from CP0
	or	    $26,    $26,	$17         # merge the two values
	mtc0	$26,    $12			        # setup new SR to CP0

	lw	    $1,     (UZ_AT*4)($27)		
	lw	    $2,	    (UZ_V0*4)($27)
	lw	    $3,	    (UZ_V1*4)($27)
	lw	    $4,	    (UZ_A0*4)($27)
	lw	    $5,     (UZ_A1*4)($27)
	lw	    $6,     (UZ_A2*4)($27)
	lw	    $7,	    (UZ_A3*4)($27)
	lw	    $8,     (UZ_T0*4)($27)
	lw	    $9,	    (UZ_T1*4)($27)
	lw	    $10,    (UZ_T2*4)($27)
	lw	    $11,    (UZ_T3*4)($27)
	lw	    $12,    (UZ_T4*4)($27)
	lw	    $13,    (UZ_T5*4)($27)
	lw	    $14,    (UZ_T6*4)($27)
	lw	    $15,    (UZ_T7*4)($27)
	lw	    $16,    (UZ_S0*4)($27)
	lw	    $17,    (UZ_S1*4)($27)
	lw	    $18,	(UZ_S2*4)($27)
	lw	    $19,    (UZ_S3*4)($27)
	lw	    $20,	(UZ_S4*4)($27)
	lw	    $21,	(UZ_S5*4)($27)
	lw	    $22,	(UZ_S6*4)($27)
	lw	    $23,	(UZ_S7*4)($27)
	lw	    $24,    (UZ_T8*4)($27)
	lw	    $25,    (UZ_T9*4)($27)	
	lw	    $28,	(UZ_GP*4)($27)
	lw	    $30,	(UZ_S8*4)($27)
	lw	    $31,	(UZ_RA*4)($27)

	lw	    $26,    (UZ_MODE*4)($27)    
    mtc2    $26,    $1                  # restore CP2_MODE from uzone

# -----------------------------------------------------------------------------------
# eret function
# -----------------------------------------------------------------------------------

hal_kentry_eret:
	nop
    eret

    .set reorder
    .set at

#------------------------------------------------------------------------------------

