/*
 * chdev.c - channel device descriptor operations implementation.
 *
 * Authors  Alain Greiner   (2016)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH.is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH.is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH.; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <kernel_config.h>
#include <hal_types.h>
#include <hal_special.h>
#include <hal_irqmask.h>
#include <printk.h>
#include <boot_info.h>
#include <xlist.h>
#include <kmem.h>
#include <scheduler.h>
#include <thread.h>
#include <rpc.h>
#include <chdev.h>
#include <devfs.h>


extern chdev_directory_t    chdev_dir;   // allocated in kernel_init.c

#if CONFIG_READ_DEBUG
extern uint32_t enter_chdev_cmd;
extern uint32_t exit_chdev_cmd;
extern uint32_t enter_chdev_server;
extern uint32_t exit_chdev_server;
#endif

////////////////////////////////////////////
char * chdev_func_str( uint32_t func_type ) 
{
  	if     ( func_type == DEV_FUNC_RAM ) return "RAM";
  	else if( func_type == DEV_FUNC_ROM ) return "ROM";
  	else if( func_type == DEV_FUNC_FBF ) return "FBF";
  	else if( func_type == DEV_FUNC_IOB ) return "IOB";
  	else if( func_type == DEV_FUNC_IOC ) return "IOC";
  	else if( func_type == DEV_FUNC_MMC ) return "MMC";
  	else if( func_type == DEV_FUNC_DMA ) return "DMA";
  	else if( func_type == DEV_FUNC_NIC ) return "NIC";
  	else if( func_type == DEV_FUNC_TIM ) return "TIM";
  	else if( func_type == DEV_FUNC_TXT ) return "TXT";
  	else if( func_type == DEV_FUNC_ICU ) return "ICU";
  	else if( func_type == DEV_FUNC_PIC ) return "PIC";
    else                                 return "undefined";
}

/////////////////////////////////////////
chdev_t * chdev_create( uint32_t    func,
                        uint32_t    impl,
                        uint32_t    channel,
                        uint32_t    is_rx,
                        xptr_t      base )
{
    chdev_t    * chdev;
    kmem_req_t   req;

    // allocate memory for chdev
    req.type   = KMEM_DEVICE;
    req.flags  = AF_ZERO;
    chdev      = (chdev_t *)kmem_alloc( &req );

    if( chdev == NULL ) return NULL;

    // initialize waiting threads queue and associated lock
    remote_spinlock_init( XPTR( local_cxy , &chdev->wait_lock ) );
    xlist_root_init( XPTR( local_cxy , &chdev->wait_root ) );

    // initialize attributes
    chdev->func    =  func;
    chdev->impl    =  impl;
    chdev->channel =  channel;
    chdev->is_rx   =  is_rx;
    chdev->base    =  base; 

    return chdev;

}  // end chdev_create()

///////////////////////////////////
void chdev_print( chdev_t * chdev )
{
    printk("\n - func      = %s"
           "\n - channel   = %d"
           "\n - base      = %l"
           "\n - cmd       = %x"
           "\n - isr       = %x"
           "\n - chdev     = %x\n",
           chdev_func_str(chdev->func),
           chdev->channel,
           chdev->base,
           chdev->cmd,
           chdev->isr,
           chdev );
}

//////////////////////////////////////////////////
void chdev_register_command( xptr_t     chdev_xp )
{
    thread_t * server_ptr;    // local pointer on server thread associated to chdev
    core_t   * core_ptr;      // local pointer on core running the server thread
    uint32_t   lid;           // core running the server thread local index
    xptr_t     lock_xp;       // extended pointer on lock protecting the chdev queue
    uint32_t   different;     // non zero if server thread core != client thread core
    uint32_t   save_sr;       // for critical section

    thread_t * this = CURRENT_THREAD;

chdev_dmsg("\n[DBG] %s : core[%x,%d] (thread %s) enter / cycle %d\n",
__FUNCTION__, local_cxy, this->core->lid, thread_type_str(this->type) , hal_time_stamp() );

    // get device descriptor cluster and local pointer
    cxy_t     chdev_cxy = GET_CXY( chdev_xp );
    chdev_t * chdev_ptr = (chdev_t *)GET_PTR( chdev_xp );

    // build extended pointers on client thread xlist and device root
    xptr_t  list_xp = XPTR( local_cxy , &this->wait_list );
    xptr_t  root_xp = XPTR( chdev_cxy , &chdev_ptr->wait_root );

    // get local pointer on server thread
    server_ptr = (thread_t *)hal_remote_lpt( XPTR( chdev_cxy , &chdev_ptr->server) );

chdev_dmsg("\n[DBG] %s : core[%x,%d] (thread %s) / server_cxy %x / server_ptr %x / server_type %\n",
__FUNCTION__, local_cxy, this->core->lid, server_cxy, server_ptr,
thread_type_str( hal_remote_lw( XPTR( server_cxy , &server_ptr->type) ) ) );

    // build extended pointer on chdev lock protecting queue
    lock_xp = XPTR( chdev_cxy , &chdev_ptr->wait_lock );

    // get local pointer on core running the server thread
    core_ptr = (core_t *)hal_remote_lpt( XPTR( chdev_cxy , &server_ptr->core ) );

    // get core local index
    lid = hal_remote_lw( XPTR( chdev_cxy , &core_ptr->lid ) );

    // compute server core != thread core
    different = (lid != this->core->lid) || (local_cxy != chdev_cxy);

    // enter critical section to make atomic : 
    // (1) client blocking
    // (2) client registration in server queue 
    // (3) IPI to force server scheduling
    // (4) descheduling 
    // ... in this order
    hal_disable_irq( &save_sr );

    // block current thread
    thread_block( CURRENT_THREAD , THREAD_BLOCKED_IO );

    // register client thread in waiting queue 
    remote_spinlock_lock( lock_xp );
    xlist_add_last( root_xp , list_xp );
    remote_spinlock_unlock( lock_xp );

    // send IPI to core running the server thread if required
    if( different ) dev_pic_send_ipi( chdev_cxy , lid ); 
    
chdev_dmsg("\n[DBG] %s : core[%x,%d] (thread %s) deschedules / cycle %d\n",
__FUNCTION__, local_cxy, this->core->lid, thread_type_str(this->type) , hal_time_stamp() );

    // deschedule
    assert( thread_can_yield( this ) , __FUNCTION__ , "illegal sched_yield\n" );
    sched_yield("blocked on I/O");

chdev_dmsg("\n[DBG] %s : core[%x,%d] (thread %s) resumes / cycle %d\n",
__FUNCTION__, local_cxy, this->core->lid, thread_type_str(this->type) , hal_time_stamp() );

    // exit critical section
    hal_restore_irq( save_sr );

}  // end chdev_register_command()

///////////////////////////////////////////////
void chdev_sequencial_server( chdev_t * chdev )
{
    xptr_t          client_xp;    // extended pointer on waiting thread
    cxy_t           client_cxy;   // cluster of client thread
    thread_t      * client_ptr;   // local pointer on client thread
    thread_t      * server;       // local pointer on server thread
    xptr_t          root_xp;      // extended pointer on device waiting queue root
    xptr_t          lock_xp;      // extended pointer on lock ptotecting chdev queue

    server = CURRENT_THREAD;

chdev_dmsg("\n[DBG] %s : enter / server = %x / chdev = %x / cycle %d\n",
__FUNCTION__ , server , chdev , hal_time_stamp() );

    root_xp = XPTR( local_cxy , &chdev->wait_root );
    lock_xp = XPTR( local_cxy , &chdev->wait_lock );

	// This infinite loop is executed by the DEV thread
    // to handle commands registered in the chdev queue.
    while( 1 )
    {
        // get the lock protecting the waiting queue
        remote_spinlock_lock( lock_xp );

        // check waiting queue state
        if( xlist_is_empty( root_xp ) ) // waiting queue empty 
        {
            // release lock 
            remote_spinlock_unlock( lock_xp );

chdev_dmsg("\n[DBG] %s : thread %x deschedule /cycle %d\n",
__FUNCTION__ , server , hal_time_stamp() );

            // deschedule
            sched_yield("I/O queue empty");

chdev_dmsg("\n[DBG] %s : thread %x resume /cycle %d\n",
__FUNCTION__ , server , hal_time_stamp() );

        } 
        else                            // waiting queue not empty
        {

#if CONFIG_READ_DEBUG
enter_chdev_server = hal_time_stamp();
#endif
            // release lock 
            remote_spinlock_unlock( lock_xp );

            // get extended pointer on first client thread
            client_xp = XLIST_FIRST_ELEMENT( root_xp , thread_t , wait_list );

            // get client thread cluster, local pointer, and identifier 
            client_cxy = GET_CXY( client_xp );
            client_ptr = (thread_t *)GET_PTR( client_xp );

            // call driver command function to execute I/O operation
            chdev->cmd( client_xp );
        
            // remove the client thread from waiting queue
            remote_spinlock_lock( lock_xp );
            xlist_unlink( XPTR( client_cxy , &client_ptr->wait_list ) );
            remote_spinlock_unlock( lock_xp );

chdev_dmsg("\n[DBG] %s : thread %x complete operation for client %x / cycle %d\n",
__FUNCTION__ , server , client_ptr , hal_time_stamp() );

#if CONFIG_READ_DEBUG
exit_chdev_server = hal_time_stamp();
#endif

        }
    }  // end while
}  // end chdev_sequencial_server()

////////////////////////
void chdev_dir_display()
{
    cxy_t     iob_cxy  = GET_CXY( chdev_dir.iob );
    chdev_t * iob_ptr  = (chdev_t *)GET_PTR( chdev_dir.iob );
    uint32_t  iob_base = (uint32_t)hal_remote_lwd( XPTR( iob_cxy , &iob_ptr->base ) );

    cxy_t     pic_cxy  = GET_CXY( chdev_dir.pic );
    chdev_t * pic_ptr  = (chdev_t *)GET_PTR( chdev_dir.pic );
    uint32_t  pic_base = (uint32_t)hal_remote_lwd( XPTR( pic_cxy , &pic_ptr->base ) );

    cxy_t     txt0_tx_cxy  = GET_CXY( chdev_dir.txt_tx[0] );
    chdev_t * txt0_tx_ptr  = (chdev_t *)GET_PTR( chdev_dir.txt_tx[0] );
    uint32_t  txt0_tx_base = (uint32_t)hal_remote_lwd( XPTR( txt0_tx_cxy , &txt0_tx_ptr->base ) );

    cxy_t     txt0_rx_cxy  = GET_CXY( chdev_dir.txt_rx[0] );
    chdev_t * txt0_rx_ptr  = (chdev_t *)GET_PTR( chdev_dir.txt_rx[0] );
    uint32_t  txt0_rx_base = (uint32_t)hal_remote_lwd( XPTR( txt0_rx_cxy , &txt0_rx_ptr->base ) );

    cxy_t     txt1_tx_cxy  = GET_CXY( chdev_dir.txt_tx[1] );
    chdev_t * txt1_tx_ptr  = (chdev_t *)GET_PTR( chdev_dir.txt_tx[1] );
    uint32_t  txt1_tx_base = (uint32_t)hal_remote_lwd( XPTR( txt1_tx_cxy , &txt1_tx_ptr->base ) );

    cxy_t     txt1_rx_cxy  = GET_CXY( chdev_dir.txt_rx[1] );
    chdev_t * txt1_rx_ptr  = (chdev_t *)GET_PTR( chdev_dir.txt_rx[1] );
    uint32_t  txt1_rx_base = (uint32_t)hal_remote_lwd( XPTR( txt1_rx_cxy , &txt1_rx_ptr->base ) );

    cxy_t     txt2_tx_cxy  = GET_CXY( chdev_dir.txt_tx[2] );
    chdev_t * txt2_tx_ptr  = (chdev_t *)GET_PTR( chdev_dir.txt_tx[2] );
    uint32_t  txt2_tx_base = (uint32_t)hal_remote_lwd( XPTR( txt2_tx_cxy , &txt2_tx_ptr->base ) );

    cxy_t     txt2_rx_cxy  = GET_CXY( chdev_dir.txt_rx[2] );
    chdev_t * txt2_rx_ptr  = (chdev_t *)GET_PTR( chdev_dir.txt_rx[2] );
    uint32_t  txt2_rx_base = (uint32_t)hal_remote_lwd( XPTR( txt2_rx_cxy , &txt2_rx_ptr->base ) );

    cxy_t     ioc_cxy  = GET_CXY( chdev_dir.ioc[0] );
    chdev_t * ioc_ptr  = (chdev_t *)GET_PTR( chdev_dir.ioc[0] );
    uint32_t  ioc_base = (uint32_t)hal_remote_lwd( XPTR( ioc_cxy , &ioc_ptr->base ) );

    cxy_t     fbf_cxy  = GET_CXY( chdev_dir.fbf[0] );
    chdev_t * fbf_ptr  = (chdev_t *)GET_PTR( chdev_dir.fbf[0] );
    uint32_t  fbf_base = (uint32_t)hal_remote_lwd( XPTR( fbf_cxy , &fbf_ptr->base ) );

    cxy_t     nic0_rx_cxy  = GET_CXY( chdev_dir.nic_rx[0] );
    chdev_t * nic0_rx_ptr  = (chdev_t *)GET_PTR( chdev_dir.nic_rx[0] );
    uint32_t  nic0_rx_base = (uint32_t)hal_remote_lwd( XPTR( nic0_rx_cxy , &nic0_rx_ptr->base ) );

    cxy_t     nic0_tx_cxy  = GET_CXY( chdev_dir.nic_tx[0] );
    chdev_t * nic0_tx_ptr  = (chdev_t *)GET_PTR( chdev_dir.nic_tx[0] );
    uint32_t  nic0_tx_base = (uint32_t)hal_remote_lwd( XPTR( nic0_tx_cxy , &nic0_tx_ptr->base ) );

    printk("\n***** external chdev directory in cluster %x\n"
           "  - iob       : cxy = %X / ptr = %X / base = %X\n"
           "  - pic       : cxy = %X / ptr = %X / base = %X\n"
           "  - ioc       : cxy = %X / ptr = %X / base = %X\n"
           "  - fbf       : cxy = %X / ptr = %X / base = %X\n"
           "  - txt_rx[0] : cxy = %X / ptr = %X / base = %X\n"
           "  - txt_tx[0] : cxy = %X / ptr = %X / base = %X\n"
           "  - txt_rx[1] : cxy = %X / ptr = %X / base = %X\n"
           "  - txt_tx[1] : cxy = %X / ptr = %X / base = %X\n"
           "  - txt_rx[2] : cxy = %X / ptr = %X / base = %X\n"
           "  - txt_tx[2] : cxy = %X / ptr = %X / base = %X\n"
           "  - nic_rx[0] : cxy = %X / ptr = %X / base = %X\n"
           "  - nic_tx[0] : cxy = %X / ptr = %X / base = %X\n",
           local_cxy,
           iob_cxy , iob_ptr , iob_base ,
           pic_cxy , pic_ptr , pic_base ,
           ioc_cxy , ioc_ptr , ioc_base ,
           fbf_cxy , fbf_ptr , fbf_base ,
           txt0_rx_cxy , txt0_rx_ptr , txt0_rx_base ,
           txt0_tx_cxy , txt0_tx_ptr , txt0_tx_base ,
           txt1_rx_cxy , txt1_rx_ptr , txt1_rx_base ,
           txt1_tx_cxy , txt1_tx_ptr , txt1_tx_base ,
           txt2_rx_cxy , txt2_rx_ptr , txt2_rx_base ,
           txt2_tx_cxy , txt2_tx_ptr , txt2_tx_base ,
           nic0_rx_cxy , nic0_rx_ptr , nic0_rx_base ,
           nic0_tx_cxy , nic0_tx_ptr , nic0_tx_base );

}  // end chdev_dir_display()

