/*
 * scheduler.h - Core scheduler definition.
 * 
 * Author    Alain Greiner (2016)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef _SCHEDULER_H_
#define _SCHEDULER_H_

#include <hal_types.h>
#include <list.h>
#include <spinlock.h>

/****  Forward declarations  ****/

struct core_s;
struct thread_s;

/*********************************************************************************************
 * This structure define the scheduler associated to a given core.
 ********************************************************************************************/

typedef struct scheduler_s
{
    spinlock_t        lock;         /*! readlock protecting lists of threads                */
    uint16_t          u_threads_nr; /*! total numbre of attached user threads               */
    uint16_t          k_threads_nr; /*! total number of attached kernel threads             */
    list_entry_t      u_root;       /*! root of list of user threads for this scheduler     */
    list_entry_t      k_root;       /*! root of list of kernel threads for this scheduler   */
    list_entry_t    * u_last;       /*! pointer on list_entry for last executed k_thread    */
    list_entry_t    * k_last;       /*! pointer on list entry for last executed u_thread    */
    struct thread_s * idle;         /*! pointer on idle thread                              */
    struct thread_s * current;      /*! pointer on current running thread                   */
}
scheduler_t;

/*********************************************************************************************
 *  This function initialises the scheduler for a given core.
 ********************************************************************************************/  
void sched_init( struct core_s * core );

/*********************************************************************************************
 * This function register a new thread in a given core scheduler.
 *********************************************************************************************
 * @ core    : local pointer on the core descriptor.
 * @ thread  : local pointer on the thread descriptor.
 ********************************************************************************************/  
void sched_register_thread( struct core_s   * core,
                            struct thread_s * thread );

/********************************************************************************************* 
 *  This function removes a thread from the set of threads attached to a given core.
 *********************************************************************************************
 * @ thread  : local pointer on the thread descriptor.
 ********************************************************************************************/  
void sched_remove_thread( struct thread_s * thread );

/********************************************************************************************* 
 * This function is the only method to make a context switch. It is called in cas of TICK,
 * or when when a thread explicitely requires a scheduling.
 * It handles the pending signals for all threads attached to the core running the calling
 * thread, and calls the sched_select() function to select a new thread.
 * The cause argument is only used for debug by the sched_display() function, and 
 * indicates the scheduling cause.
 *********************************************************************************************
 * @ cause    : character string defining the scheduling cause.
 ********************************************************************************************/
void sched_yield( char * cause );

/*********************************************************************************************
 * This function scan all threads attached to a given core scheduler, and executes
 * the relevant actions for pending signals, such as the THREAD_SIG_KILL signal.
 *********************************************************************************************
 * @ core    : local pointer on the core descriptor.
 ********************************************************************************************/
void sched_handle_signals( struct core_s * core );

/********************************************************************************************* 
 * This function is used by the scheduler of a given core to actually kill a thread that has 
 * the SIG_KILL / SIG_SUICIDE signal set (following a thread_exit() or a thread_kill() event).
 * - It checks that the thread has released all locks => panic otherwise...
 * - It removes the thread from the scheduler.
 * - It reset the SIG_KILL signal to acknoledge the killer.
 * - In case of SIG_SUCIDE, it removes the detached thread from its process, and destroys it.
 *********************************************************************************************
 * @ thread  : local pointer on the thread descriptor.
 ********************************************************************************************/
void sched_kill_thread( struct thread_s * thread );

/*********************************************************************************************
 * This function does NOT modify the scheduler state.
 * It just select a thread in the list of attached threads, implementing the following 
 * three steps policy: 
 * 1) It scan the list of kernel threads, from the next thread after the last executed one,
 *    and returns the first runnable found : not IDLE, not blocked, client queue not empty.
 *    It can be the current thread.
 * 2) If no kernel thread found, it scan the list of user thread, from the next thread after
 *    the last executed one, and returns the first runable found : not blocked.
 *    It can be the current thread.
 * 3) If no runable thread found, it returns the idle thread.
 *********************************************************************************************
 * @ core    : local pointer on scheduler.
 * @ returns pointer on selected thread descriptor
 ********************************************************************************************/
struct thread_s * sched_select( struct scheduler_s * sched );

/********************************************************************************************* 
 * This function display the internal state of the local core identified by its <lid>.
 *********************************************************************************************
 * @ lid      : local index of target core.
 ********************************************************************************************/
void sched_display( lid_t lid );


#endif	/* _SCHEDULER_H_ */
