/*
 * sys_exit.c - Kernel function implementing the "exit" system call.
 * 
 * Author    Alain Greiner (2016,2017)
 *
 * Copyright (c)  UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <kernel_config.h>
#include <hal_types.h>
#include <hal_irqmask.h>
#include <errno.h>
#include <thread.h>
#include <printk.h>
#include <process.h>
#include <signal.h>
#include <cluster.h>
#include <rpc.h>

///////////////////////////////
int sys_exit( uint32_t status )
{
    uint32_t    save_sr;       // required to enable IRQs

    thread_t  * this = CURRENT_THREAD;
    pid_t       pid  = this->process->pid;

#if CONFIG_SYSCALL_DEBUG
uint64_t    tm_start;
uint64_t    tm_end;
tm_start = hal_get_cycles();
printk("\n[DBG] %s : core[%x,%d] enter / process %x / status %x / cycle %d\n",
__FUNCTION__ , local_cxy , this->core->lid , pid , status , (uint32_t)tm_start );
#endif

    // get owner process cluster 
    cxy_t   owner_cxy  = CXY_FROM_PID( pid );

    // enable IRQs
    hal_enable_irq( &save_sr );

    // execute process_make_exit() function in owner cluster
    if( local_cxy == owner_cxy )                                // owner is local 
    {
        process_make_exit( pid , status );
    }
    else                                                        // owner is remote
    {
        rpc_process_make_exit_client( owner_cxy, pid , status );
    }

    // restore IRQs
    hal_restore_irq( save_sr );

    hal_fence();

#if CONFIG_SYSCALL_DEBUG
tm_end = hal_get_cycles();
printk("\n[DBG] %s : core[%x,%d] exit / process %x / status %x / cost = %d\n",
__FUNCTION__ , local_cxy , this->core->lid , pid , status , (uint32_t)(tm_end - tm_start) );
#endif

	return 0;

}  // end sys_exit()

