/*
 * sys_exit.c - Kernel function implementing the "exit" system call.
 * 
 * Author    Alain Greiner (2016,2017)
 *
 * Copyright (c)  UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <kernel_config.h>
#include <hal_types.h>
#include <hal_irqmask.h>
#include <errno.h>
#include <thread.h>
#include <printk.h>
#include <process.h>
#include <signal.h>
#include <cluster.h>
#include <rpc.h>

///////////////////////////////
int sys_exit( uint32_t status )
{
    reg_t       save_sr;       // required to enable IRQs

    thread_t  * this    = CURRENT_THREAD;
    process_t * process = this->process;
    pid_t       pid     = process->pid;

#if CONFIG_DEBUG_SYS_EXIT
uint64_t    tm_start;
uint64_t    tm_end;
tm_start = hal_get_cycles();
if( CONFIG_DEBUG_SYS_EXIT < tm_start )
printk("\n[DBG] %s : thread %x enter / process %x / status %x / cycle %d\n",
__FUNCTION__ , this, pid , status , (uint32_t)tm_start );
#endif

    // get cluster and pointers on process in owner cluster
    xptr_t      owner_xp  = cluster_get_owner_process_from_pid( pid );
    cxy_t       owner_cxy = GET_CXY( owner_xp );
    process_t * owner_ptr = GET_PTR( owner_xp );

    assert( (owner_xp != XPTR_NULL) , __FUNCTION__ , "owner_xp cannot be NULL\n" );

    // enable IRQs
    hal_enable_irq( &save_sr );

    // the process_make_kill() function must be executed
    // by an RPC thread in reference cluster
    rpc_process_make_kill_client( owner_cxy, owner_ptr, true , status );

    // restore IRQs
    hal_restore_irq( save_sr );

    hal_fence();

#if CONFIG_DEBUG_SYS_EXIT
tm_end = hal_get_cycles();
if( CONFIG_DEBUG_SYS_EXIT < tm_end )
printk("\n[DBG] %s : thread %x exit / process %x / status %x / cost = %d / cycle %d\n",
__FUNCTION__, this, pid, status, (uint32_t)(tm_end - tm_start), (uint32_t)tm_end );
#endif

	return 0;

}  // end sys_exit()

