/*
 * soclib_tty.c - soclib tty driver implementation.
 *
 * Author  Alain Greiner (2016)
 *
 * Copyright (c)  UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH..
 *
 * ALMOS-MKH. is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH. is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH.; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <dev_txt.h>
#include <chdev.h>
#include <soclib_tty.h>
#include <remote_spinlock.h>
#include <thread.h>
#include <printk.h>
#include <hal_special.h>

#if (CONFIG_DEBUG_SYS_READ & 1)
extern uint32_t  enter_tty_cmd_read;
extern uint32_t  exit_tty_cmd_read;

extern uint32_t  enter_tty_isr_read;
extern uint32_t  exit_tty_isr_read;
#endif

#if (CONFIG_DEBUG_SYS_WRITE & 1)
extern uint32_t  enter_tty_cmd_write;
extern uint32_t  exit_tty_cmd_write;

extern uint32_t  enter_tty_isr_write;
extern uint32_t  exit_tty_isr_write;
#endif

///////////////////////////////////////
void soclib_tty_init( chdev_t * chdev )
{
    xptr_t reg_xp;

    chdev->cmd = &soclib_tty_cmd;
    chdev->isr = &soclib_tty_isr;
    chdev->aux = &soclib_tty_aux;

    // get TTY channel and extended pointer on TTY peripheral base address
    xptr_t   tty_xp  = chdev->base;
    uint32_t channel = chdev->channel;

    // get SOCLIB_TTY device cluster and local pointer
    cxy_t      tty_cxy = GET_CXY( tty_xp );
    uint32_t * tty_ptr = GET_PTR( tty_xp );

    // set TTY_RX_IRQ_ENABLE 
    reg_xp = XPTR( tty_cxy , tty_ptr + (channel * TTY_SPAN) + TTY_RX_IRQ_ENABLE );
    hal_remote_sw( reg_xp , 1 );

    // reset TTY_TX_IRQ_ENABLE
    reg_xp = XPTR( tty_cxy , tty_ptr + (channel * TTY_SPAN) + TTY_TX_IRQ_ENABLE );
    hal_remote_sw( reg_xp , 0 );
}

//////////////////////////////////////////////////////////////
void __attribute__ ((noinline)) soclib_tty_cmd( xptr_t th_xp )
{

#if (CONFIG_DEBUG_SYS_READ & 1)
if( type == TXT_READ) enter_tty_cmd_read = (uint32_t)hal_get_cycles();
#endif

#if (CONFIG_DEBUG_SYS_WRITE & 1)
if( type == TXT_WRITE) enter_tty_cmd_write = (uint32_t)hal_get_cycles();
#endif

    // get client thread cluster and local pointer
    cxy_t      th_cxy = GET_CXY( th_xp );
    thread_t * th_ptr = GET_PTR( th_xp );

    // get command type and extended pointer on TXT device
    uint32_t type = hal_remote_lw ( XPTR( th_cxy , &th_ptr->txt_cmd.type ) );

#if CONFIG_DEBUG_HAL_TXT
uint32_t cycle = (uint32_t)hal_get_cycles();
if (CONFIG_DEBUG_HAL_TXT < cycle )
printk("\n[DBG] %s : thread %x enter for %s / cycle %d\n",
__FUNCTION__ , CURRENT_THREAD , dev_txt_type_str(type) , cycle );
#endif

    if( type == TXT_WRITE )         // block, enable TX_IRQ, and dechedule for a WRITE
    {
        xptr_t dev_xp = (xptr_t)hal_remote_lwd( XPTR( th_cxy , &th_ptr->txt_cmd.dev_xp ) );

        // get TXT device cluster and local pointer
        cxy_t     dev_cxy = GET_CXY( dev_xp );
        chdev_t * dev_ptr = GET_PTR( dev_xp );

        // get extended pointer on SOCLIB_TTY base segment
        xptr_t tty_xp = (xptr_t)hal_remote_lwd( XPTR( dev_cxy , &dev_ptr->base ) );

        // get SOCLIB_TTY base segment cluster and local pointer
        cxy_t      tty_cxy = GET_CXY( tty_xp );
        uint32_t * tty_ptr = GET_PTR( tty_xp );

        // get TTY channel index and channel base address
        uint32_t   channel = hal_remote_lw( XPTR( dev_cxy , &dev_ptr->channel ) );
        uint32_t * base    = tty_ptr + TTY_SPAN * channel;

        // block server thread 
        thread_block( CURRENT_THREAD , THREAD_BLOCKED_DEV_ISR );

        // enable relevant TX_IRQ for a WRITE
        hal_remote_sw( XPTR( tty_cxy , base + TTY_TX_IRQ_ENABLE ) , 1 );

        // deschedule
        sched_yield( "waiting TXT_TX_ISR completion" );
    }
    else if( type == TXT_READ )        // block, and deschedule for a READ
    {
        // block server thread
        thread_block( CURRENT_THREAD , THREAD_BLOCKED_DEV_ISR );

        // deschedule
        sched_yield( "waiting TXT_RX_ISR completion" );
    }
    else
    {
        assert( false , __FUNCTION__ , "illegal TXT command\n" );
    }

#if CONFIG_DEBUG_HAL_TXT
cycle = (uint32_t)hal_get_cycles();
if (CONFIG_DEBUG_HAL_TXT < cycle )
printk("\n[DBG] %s : thread %x exit after %s / cycle %d\n", 
__FUNCTION__ , CURRENT_THREAD , dev_txt_type_str( type ) , cycle );
#endif

#if (CONFIG_DEBUG_SYS_READ & 1)
if( type == TXT_READ ) exit_tty_cmd_read = (uint32_t)hal_get_cycles();
#endif

#if (CONFIG_DEBUG_SYS_WRITE & 1)
if( type == TXT_WRITE ) exit_tty_cmd_write = (uint32_t)hal_get_cycles();
#endif

}  // end soclib_tty_cmd()

/////////////////////////////////////////////////////////////
void __attribute__ ((noinline)) soclib_tty_aux( void * args )
{
    uint32_t   status;
    bool_t     empty;
    uint32_t   i;

    xptr_t     dev_xp = ((txt_sync_args_t *)args)->dev_xp;
    char     * buffer = ((txt_sync_args_t *)args)->buffer;
    uint32_t   count  = ((txt_sync_args_t *)args)->count;
    
    // get TXT0 chdev cluster and local pointer
    cxy_t     dev_cxy = GET_CXY( dev_xp );
    chdev_t * dev_ptr = (chdev_t *)GET_PTR( dev_xp );

    // get extended pointer on TTY channel base address
    xptr_t tty_xp = (xptr_t)hal_remote_lwd( XPTR( dev_cxy , &dev_ptr->base ) );

    // get TTY channel segment cluster and local pointer
    cxy_t      tty_cxy = GET_CXY( tty_xp );
    uint32_t * tty_ptr = (uint32_t *)GET_PTR( tty_xp );

    // get extended pointers on TTY_WRITE & TTY_STATUS registers
    xptr_t write_xp  = XPTR( tty_cxy , tty_ptr + TTY_WRITE );
    xptr_t status_xp = XPTR( tty_cxy , tty_ptr + TTY_STATUS );

    // loop on characters (busy waiting strategy)
    for( i = 0 ; i < count ; i++ )
    {
        do
        {
            // get TTY_STATUS
            status = hal_remote_lw( status_xp );
            empty  = ( (status & TTY_STATUS_TX_FULL) == 0 );

            // transfer one byte if TX buffer empty
            if ( empty )  hal_remote_sb( write_xp , buffer[i] );
        }
        while ( empty == false );
    }
}  // end soclib_tty_aux()


/////////////////////////////////////////////////////////////////
void __attribute__ ((noinline)) soclib_tty_isr( chdev_t * chdev )
{
    uint32_t    type;         // command type
    uint32_t    count;        // number of bytes in buffer
    xptr_t      buf_xp;       // extended pointer on buffer
    xptr_t      error_xp;     // extended pointer on error field in command
    xptr_t      status_xp;    // extended pointer on TTY_STATUS register
    xptr_t      write_xp;     // extended pointer on TTY_WRITE register
    xptr_t      read_xp;      // extended pointer on TTY_READ register
    uint32_t    status;       // TTY terminal status
    char        byte;         // read byte
    xptr_t      client_xp;    // first client thread in waiting queue
    cxy_t       client_cxy;   // firts client thread cluster
    thread_t  * client_ptr;   // first client thread pointer
    pid_t       pid;          // foreground process identifier
    xptr_t      owner_xp;     // extended pointer on foreground process in owner cluster
    cxy_t       owner_cxy;   
    process_t * owner_ptr;
    uint32_t    i;

#if (CONFIG_DEBUG_SYS_READ & 1)
enter_tty_isr_read = (uint32_t)hal_get_cycles();
#endif

#if (CONFIG_DEBUG_SYS_WRITE & 1)
enter_tty_isr_write = (uint32_t)hal_get_cycles();
#endif

#if CONFIG_DEBUG_HAL_TXT
uint32_t cycle = (uint32_t)hal_get_cycles();
if (CONFIG_DEBUG_HAL_TXT < cycle)
printk("\n[DBG] %s : enter / cycle %d\n", __FUNCTION__ , cycle );
#endif

    // get extended pointer on chdev queue root
    xptr_t root_xp   = XPTR( local_cxy , &chdev->wait_root );

    // get chdev channel 
    uint32_t channel = chdev->channel;

    // get first command if queue non empty
    if( xlist_is_empty( root_xp ) == false )
    {
        // get extended pointer on first client thread
        client_xp = XLIST_FIRST_ELEMENT( root_xp , thread_t , wait_list );

        // get client thread cluster and local pointer
        client_cxy = GET_CXY( client_xp );
        client_ptr = GET_PTR( client_xp );

        // get command arguments
        type     = hal_remote_lw ( XPTR( client_cxy , &client_ptr->txt_cmd.type   ) );
        count    = hal_remote_lw ( XPTR( client_cxy , &client_ptr->txt_cmd.count  ) );
        buf_xp   = hal_remote_lwd( XPTR( client_cxy , &client_ptr->txt_cmd.buf_xp ) );
        error_xp =                 XPTR( client_cxy , &client_ptr->txt_cmd.error  );
    }
    else
    {
        type     = 0xFFFFFFFF;

        // these lines to avoid a GCC warning
        count      = 0;
        buf_xp     = XPTR_NULL;
        error_xp   = XPTR_NULL;
        client_cxy = 0;
        client_ptr = NULL;
    }

    // get SOCLIB_TTY peripheral cluster and local pointer
    cxy_t      tty_cxy = GET_CXY( chdev->base );
    uint32_t * tty_ptr = GET_PTR( chdev->base );

    // get channel base address
    uint32_t * base = tty_ptr + TTY_SPAN * chdev->channel;

    // get extended pointer on TTY registers
    status_xp = XPTR( tty_cxy , base + TTY_STATUS );
    write_xp  = XPTR( tty_cxy , base + TTY_WRITE );
    read_xp   = XPTR( tty_cxy , base + TTY_READ );

    // get TTY_STATUS register value
    status = hal_remote_lw( status_xp );

    // 1. handle RX if TTY_READ buffer full
    if( status & TTY_STATUS_RX_FULL )   
    {
        // get a byte from TTY_READ / acknowledge RX_IRQ
        byte = (char)hal_remote_lb( read_xp );

        // check character value
        if( byte == 0x1A )          // ^Z  =>  stop onwner process
        {
            // get pid of terminal owner process
            pid = process_get_txt_owner( channel );

            // get cluster and pointers on owner process descriptor
            owner_xp  = cluster_get_owner_process_from_pid( pid );
            owner_cxy = GET_CXY( owner_xp );
            owner_ptr = GET_PTR( owner_xp );

            // send stop signal to owner process
            process_sigaction( pid , BLOCK_ALL_THREADS );
                
            // atomically update owner process termination state
            hal_remote_atomic_or( XPTR( owner_cxy , &owner_ptr->term_state ) ,
                                  PROCESS_TERM_STOP );
            return;
        }
        else if( byte == 0x03 )     // ^C  => kill owner process
        {
            // get pid of terminal owner process
            pid = process_get_txt_owner( channel );

            // get cluster and pointers on owner process descriptor
            owner_xp  = cluster_get_owner_process_from_pid( pid );
            owner_cxy = GET_CXY( owner_xp );
            owner_ptr = GET_PTR( owner_xp );

            // send kill signal to owner process
            process_sigaction( pid , DELETE_ALL_THREADS );
                
            // atomically update owner process termination state
            hal_remote_atomic_or( XPTR( owner_cxy , &owner_ptr->term_state ) ,
                                  PROCESS_TERM_KILL );
            return;
        }
        else if( type == TXT_READ ) // pending TXT_READ
        {
            // write character to command buffer
            hal_remote_sb( buf_xp , byte );

            // set I/O operation status in command
            hal_remote_sw( error_xp , 0 );

            // unblock server thread
            thread_unblock( XPTR( local_cxy , chdev->server ) , THREAD_BLOCKED_DEV_ISR );
        }
    }

    // 3. handle TX if TXT_WRITE
    if( type == TXT_WRITE )
    {
        // loop on characters
        for( i = 0 ; i < count ; i++ )
        {
            // get TTY_STATUS
            status = hal_remote_lw( status_xp );

            if( (status & TTY_STATUS_TX_FULL) == 0 ) // TTY_TX empty => move one byte
            {
                // get one byte from command buffer
                byte = (char)hal_remote_lb( buf_xp + i );

                // write byte to TTY_WRITE / acknowledge TX_IRQ
                hal_remote_sb( write_xp , byte );
            }
            else         // TTY_TX full => update command arguments and exit ISR for retry
            {
                hal_remote_sw ( XPTR( client_cxy , &client_ptr->txt_cmd.count ), count-i );
                hal_remote_swd( XPTR( client_cxy , &client_ptr->txt_cmd.buf_xp ), buf_xp+i );
                return;
            }
        }

        // set I/O operation status in command
        hal_remote_sw( error_xp , 0 );

        // disable TX_IRQ 
        hal_remote_sw( XPTR( tty_cxy , base + TTY_TX_IRQ_ENABLE ) , 0 );

        // unblock server thread
        thread_unblock( XPTR( local_cxy , chdev->server ) , THREAD_BLOCKED_DEV_ISR );
    }

    hal_fence();

#if CONFIG_DEBUG_HAL_TXT
cycle = (uint32_t)hal_get_cycles();
if (CONFIG_DEBUG_HAL_TXT < cycle)
printk("\n[DBG] %s : exit after %s / cycle %d\n",
__FUNCTION__ , dev_txt_type_str( type ) , cycle );
#endif

#if (CONFIG_DEBUG_SYS_READ & 1)
if( type == TXT_READ) exit_tty_isr_read = (uint32_t)hal_get_cycles();
#endif

#if (CONFIG_DEBUG_SYS_WRITE & 1)
if( type == TXT_WRITE) exit_tty_isr_write = (uint32_t)hal_get_cycles();
#endif

}  // end soclib_tty_isr()

