/*
 * sys_kill.c - Kernel function implementing the "kill" system call.
 * 
 * Author    Alain Greiner (2016,2017)
 *
 * Copyright (c)  UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <kernel_config.h>
#include <hal_types.h>
#include <hal_irqmask.h>
#include <errno.h>
#include <thread.h>
#include <printk.h>
#include <process.h>
#include <shared_syscalls.h>
#include <cluster.h>
#include <rpc.h>

///////////////////////////
int sys_kill( pid_t    pid,
              uint32_t sig_id )
{
    uint32_t    save_sr;       // required to enable IRQs
    xptr_t      owner_xp;      // extended pointer on target reference process
    cxy_t       owner_cxy;     // target process cluster
    process_t * owner_ptr;     // local pointer on target process
    xptr_t      parent_xp;     // extended pointer on parent process
    cxy_t       parent_cxy;    // parent process cluster
    process_t * parent_ptr;    // local pointer on parent process
    pid_t       ppid;          // parent process PID
    uint32_t    retval;        // return value for the switch

    thread_t  * this    = CURRENT_THREAD;
    process_t * process = this->process;

#if CONFIG_DEBUG_SYS_KILL
uint64_t    tm_start;
uint64_t    tm_end;
tm_start = hal_get_cycles();
if( CONFIG_DEBUG_SYS_KILL < tm_start )
printk("\n[DBG] %s : thread %x enter / process %x / sig %d / cycle %d\n",
__FUNCTION__ , this, pid, sig_id, (uint32_t)tm_start );
#endif

    // process cannot kill itself
    if( pid == process->pid )
    {

#if CONFIG_DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : process %d cannot kill itself\n", __FUNCTION__ , pid );
#endif
        this->errno = EINVAL;
        return -1;
    }

    // get cluster and pointers on owner target process descriptor
    owner_xp  = cluster_get_owner_process_from_pid( pid );
    owner_cxy = GET_CXY( owner_xp );
    owner_ptr = GET_PTR( owner_xp );

    // check process existence
    if( owner_xp == XPTR_NULL )
    {

#if CONFIG_DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : process %x not found\n", __FUNCTION__ , pid );
#endif
        this->errno = EINVAL;
        return -1;
    }
    
    // get parent process PID
    parent_xp  = hal_remote_lwd( XPTR( owner_cxy , &owner_ptr->parent_xp ) );
    parent_cxy = GET_CXY( parent_xp );
    parent_ptr = GET_PTR( parent_xp );
    ppid       = hal_remote_lw( XPTR( parent_cxy , &parent_ptr->pid ) );

    // processes INIT 
    if( pid == 1 )
    {

#if CONFIG_DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : process_init cannot be killed\n", __FUNCTION__ );
#endif
		this->errno = EINVAL;
        return -1;
    }

    // enable IRQs
    hal_enable_irq( &save_sr );

    // analyse signal type / supported values are : 0, SIGSTOP, SIGCONT, SIGKILL
    switch( sig_id )
    {
        case 0 :
        {
            // does nothing
            retval = 0;
            break;
        }
        case SIGSTOP:     
        {
            // remove TXT ownership from target process 
            process_txt_reset_ownership( owner_xp );

            // block all threads in all clusters
            process_sigaction( pid , BLOCK_ALL_THREADS );

            // atomically update owner process termination state
            hal_remote_atomic_or( XPTR( owner_cxy , &owner_ptr->term_state ) ,
                                  PROCESS_TERM_STOP );
 
            retval = 0;
            break;
        }
        case SIGCONT:
        {
            // unblock all threads in all clusters
            process_sigaction( pid , UNBLOCK_ALL_THREADS );

            // atomically update reference process termination state
            hal_remote_atomic_and( XPTR( owner_cxy , &owner_ptr->term_state ) ,
                                   ~PROCESS_TERM_STOP );
            retval = 0;
            break;
        }
        break;
        case SIGKILL:
        {
            // remove TXT ownership from owner process descriptor
            process_txt_reset_ownership( owner_xp );

            // block all process threads in all clusters
            process_sigaction( pid , BLOCK_ALL_THREADS );

            // mark all process threads in all clusters for delete 
            process_sigaction( pid , DELETE_ALL_THREADS );

            // atomically update owner process descriptor flags
            hal_remote_atomic_or( XPTR( owner_cxy , &owner_ptr->term_state ) ,
                                  PROCESS_TERM_KILL );

            retval = 0;
            break;
        }
        default:
        {

#if CONFIG_DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : illegal signal %d / process %x\n", __FUNCTION__, sig_id, pid );
#endif
    	    this->errno = EINVAL;
            retval = -1;
            break;
        }
    }
    
    // restore IRQs
    hal_restore_irq( save_sr );

    hal_fence();

#if CONFIG_DEBUG_SYS_KILL
tm_end = hal_get_cycles();
if( CONFIG_DEBUG_SYS_KILL < tm_end )
printk("\n[DBG] %s : thread %x enter / process %x / sig %d / cost = %d / cycle %d\n",
__FUNCTION__ , this, pid, sig_id, (uint32_t)(tm_end - tm_start), (uint32_t)tm_end );
#endif

	return retval;

}  // end sys_kill()

