/*
 * sys_exit.c - Kernel function implementing the "exit" system call.
 * 
 * Author    Alain Greiner (2016,2017)
 *
 * Copyright (c)  UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <kernel_config.h>
#include <hal_types.h>
#include <hal_irqmask.h>
#include <errno.h>
#include <thread.h>
#include <printk.h>
#include <process.h>
#include <shared_syscalls.h>
#include <cluster.h>
#include <rpc.h>

///////////////////////////////
int sys_exit( uint32_t status )
{
    reg_t       save_sr;       // required to enable IRQs

    thread_t  * this    = CURRENT_THREAD;
    process_t * process = this->process;
    pid_t       pid     = process->pid;

#if CONFIG_DEBUG_SYS_EXIT
uint64_t    tm_start;
uint64_t    tm_end;
tm_start = hal_get_cycles();
if( CONFIG_DEBUG_SYS_EXIT < tm_start )
printk("\n[DBG] %s : thread %x enter / process %x / status %x / cycle %d\n",
__FUNCTION__ , this, pid , status , (uint32_t)tm_start );
#endif

    // get owner cluster
    cxy_t  owner_cxy = CXY_FROM_PID( pid );

    // exit must be called by the main thread
    if( (owner_cxy != local_cxy) || (LTID_FROM_TRDID( this->trdid ) != 0) )
    {

#if CONFIG_DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] %s must be called by thread 0 in process owner cluster\n"
       "         trdid = %x / pid = %x / local_cxy = %x\n",
__FUNCTION__, this->trdid, pid, local_cxy );
#endif
         this->errno = EINVAL;
         return -1;
    }

    // enable IRQs
    hal_enable_irq( &save_sr );

    // register exit_status in owner process descriptor
    process->term_state = status;

    // remove TXT ownership from owner process descriptor
    process_txt_reset_ownership( XPTR( local_cxy , process ) );

    // block all process threads in all clusters
    process_sigaction( pid , BLOCK_ALL_THREADS );

    // mark all process threads in all clusters for delete 
    process_sigaction( pid , DELETE_ALL_THREADS );

    // restore IRQs
    hal_restore_irq( save_sr );

    // atomically update owner process descriptor term_state
    hal_remote_atomic_or( XPTR( local_cxy , &process->term_state ) ,
                          PROCESS_TERM_EXIT );
    hal_fence();

#if CONFIG_DEBUG_SYS_EXIT
tm_end = hal_get_cycles();
if( CONFIG_DEBUG_SYS_EXIT < tm_end )
printk("\n[DBG] %s : thread %x exit / process %x / status %x / cost = %d / cycle %d\n",
__FUNCTION__, this, pid, status, (uint32_t)(tm_end - tm_start), (uint32_t)tm_end );
#endif

	return 0;

}  // end sys_exit()

