/*
 * devfs.c - DEVFS File system API implementation.
 *
 * Author   Mohamed Lamine Karaoui (2014,2015)
 *          Alain Greiner (2016,2017)
 *
 * Copyright (c) Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_types.h>
#include <hal_special.h>
#include <hal_uspace.h>
#include <printk.h>
#include <chdev.h>
#include <thread.h>
#include <dev_txt.h>
#include <cluster.h>
#include <vfs.h>
#include <kmem.h>
#include <devfs.h>

/////////////////////////////////////////////////////////////////////////////////////////
//     Extern variables
/////////////////////////////////////////////////////////////////////////////////////////

extern vfs_ctx_t            fs_context[];   // allocated in kernel_init.c
extern chdev_directory_t    chdev_dir;      // allocated in kernel_init.c

#if (CONFIG_DEBUG_SYS_READ & 1)
extern uint32_t  enter_devfs_read;
extern uint32_t  exit_devfs_read;
#endif

#if (CONFIG_DEBUG_SYS_WRITE & 1)
extern uint32_t  enter_devfs_write;
extern uint32_t  exit_devfs_write;
#endif

///////////////////////////////
devfs_ctx_t * devfs_ctx_alloc()
{
    kmem_req_t    req;

	req.type    = KMEM_DEVFS_CTX;
	req.size    = sizeof(devfs_ctx_t);
    req.flags   = AF_KERNEL | AF_ZERO;

	return (devfs_ctx_t *)kmem_alloc( &req );
}

/////////////////////////////////////////////
void devfs_ctx_init( devfs_ctx_t * devfs_ctx,
                     xptr_t        devfs_dev_inode_xp,
                     xptr_t        devfs_external_inode_xp )
{
    devfs_ctx->dev_inode_xp      = devfs_dev_inode_xp;
    devfs_ctx->external_inode_xp = devfs_external_inode_xp;

    fs_context[FS_TYPE_DEVFS].extend = devfs_ctx;
}

/////////////////////////////////////////////////
void devfs_ctx_destroy( devfs_ctx_t * devfs_ctx )
{
    kmem_req_t    req;

    req.type = KMEM_DEVFS_CTX;
    req.ptr  = devfs_ctx;
    kmem_free( &req );
}

///////////////////////////////////////////////////
void devfs_global_init( xptr_t   parent_inode_xp,
                        xptr_t * devfs_dev_inode_xp,
                        xptr_t * devfs_external_inode_xp )
{
    error_t  error;

#if CONFIG_DEBUG_DEVFS_INIT
uint32_t cycle = (uint32_t)hal_get_cycles();
if( CONFIG_DEBUG_DEVFS_INIT < cycle )
printk("\n[DBG] %s : thread %x enter at cycle %d\n",
__FUNCTION__ , CURRENT_THREAD , cycle );
#endif

    // creates DEVFS "dev" inode in cluster IO
    error = vfs_add_child_in_parent( LOCAL_CLUSTER->io_cxy,
                                     INODE_TYPE_DIR,
                                     FS_TYPE_DEVFS,
                                     parent_inode_xp,
                                     "dev",
                                     NULL,
                                     devfs_dev_inode_xp ); 

    assert( (error == 0) , __FUNCTION__ , "cannot create <dev>\n" );

    // create DEVFS "external" inode in cluster IO
    error = vfs_add_child_in_parent( LOCAL_CLUSTER->io_cxy,
                                     INODE_TYPE_DIR,
                                     FS_TYPE_DEVFS,
                                     *devfs_dev_inode_xp,
                                     "external",
                                     NULL,
                                     devfs_external_inode_xp ); 

    assert( (error == 0) , __FUNCTION__ , "cannot create <external>\n" );

#if CONFIG_DEBUG_DEVFS_INIT
cycle = (uint32_t)hal_get_cycles();
if( CONFIG_DEBUG_DEVFS_INIT < cycle )
printk("\n[DBG] %s : thread %x exit at cycle %d\n",
__FUNCTION__ , CURRENT_THREAD , cycle );
#endif

} 

///////////////////////////////////////////////////
void devfs_local_init( xptr_t   devfs_dev_inode_xp,
                       xptr_t   devfs_external_inode_xp,
                       xptr_t * devfs_internal_inode_xp )
{
    char          node_name[16];
    xptr_t        chdev_xp;
    cxy_t         chdev_cxy;
    chdev_t     * chdev_ptr;
    xptr_t        inode_xp;
    uint32_t      channel;

#if CONFIG_DEBUG_DEVFS_INIT
uint32_t cycle = (uint32_t)hal_get_cycles();
if( CONFIG_DEBUG_DEVFS_INIT < cycle )
printk("\n[DBG] %s : thread %x enter at cycle %d\n",
__FUNCTION__ , CURRENT_THREAD , cycle );
#endif

    // create "internal" directory linked to "dev"
    snprintf( node_name , 16 , "internal_%x" , local_cxy );
    vfs_add_child_in_parent( local_cxy,
                             INODE_TYPE_DIR,
                             FS_TYPE_DEVFS,
                             devfs_dev_inode_xp,
                             node_name,
                             NULL,
                             devfs_internal_inode_xp );

    // create MMC chdev inode
    chdev_xp  = chdev_dir.mmc[local_cxy];
    if( chdev_xp != XPTR_NULL)
    {
        chdev_ptr = (chdev_t *)GET_PTR( chdev_xp );
        vfs_add_child_in_parent( local_cxy,
                                 INODE_TYPE_DEV,
                                 FS_TYPE_DEVFS,
                                 *devfs_internal_inode_xp,
                                 chdev_ptr->name,
                                 GET_PTR( chdev_xp ),
                                 &inode_xp );
    }

    // create DMA chdev inodes (one DMA channel per core)
    for( channel = 0 ; channel < LOCAL_CLUSTER->cores_nr ; channel++ )
    {
        chdev_xp = chdev_dir.dma[channel];
        if( chdev_xp != XPTR_NULL)
        {
            chdev_ptr = (chdev_t *)GET_PTR( chdev_xp );
            vfs_add_child_in_parent( local_cxy,
                                     INODE_TYPE_DEV,
                                     FS_TYPE_DEVFS,
                                     *devfs_internal_inode_xp,
                                     chdev_ptr->name,
                                     GET_PTR( chdev_xp ),
                                     &inode_xp );
        }
    }

    // create an IOB inode in cluster containing IOB chdev
    chdev_xp = chdev_dir.iob;
    if( chdev_xp != XPTR_NULL )
    {
        chdev_cxy = GET_CXY( chdev_xp );
        chdev_ptr = (chdev_t *)GET_PTR( chdev_xp );
        if( chdev_cxy == local_cxy )
        {
            vfs_add_child_in_parent( local_cxy,
                                     INODE_TYPE_DEV,
                                     FS_TYPE_DEVFS,
                                     devfs_external_inode_xp,
                                     chdev_ptr->name,
                                     GET_PTR( chdev_xp ),
                                     &inode_xp );
        }
    }
        
    // create a PIC inode in cluster containing PIC chdev
    chdev_xp = chdev_dir.pic;
    if( chdev_xp != XPTR_NULL )
    {
        chdev_cxy = GET_CXY( chdev_xp );
        chdev_ptr = (chdev_t *)GET_PTR( chdev_xp );
        if( chdev_cxy == local_cxy )
        {
            vfs_add_child_in_parent( local_cxy,
                                     INODE_TYPE_DEV,
                                     FS_TYPE_DEVFS,
                                     devfs_external_inode_xp,
                                     chdev_ptr->name,
                                     GET_PTR( chdev_xp ),
                                     &inode_xp );
        }
    }

    // create a TXT_RX inode in each cluster containing a TXT_RX chdev
    for( channel = 0 ; channel < CONFIG_MAX_TXT_CHANNELS ; channel++ )
    {
        chdev_xp = chdev_dir.txt_rx[channel];
        if( chdev_xp != XPTR_NULL )
        {
            chdev_cxy = GET_CXY( chdev_xp );
            chdev_ptr = (chdev_t *)GET_PTR( chdev_xp );
            if( chdev_cxy == local_cxy )
            {
                vfs_add_child_in_parent( local_cxy,
                                         INODE_TYPE_DEV,
                                         FS_TYPE_DEVFS,
                                         devfs_external_inode_xp,
                                         chdev_ptr->name,
                                         GET_PTR( chdev_xp ),
                                         &inode_xp );
            }
        }
    }

    // create a TXT_TX inode in each cluster containing a TXT_TX chdev
    for( channel = 0 ; channel < CONFIG_MAX_TXT_CHANNELS ; channel++ )
    {
        chdev_xp = chdev_dir.txt_tx[channel];
        if( chdev_xp != XPTR_NULL )
        {
            chdev_cxy = GET_CXY( chdev_xp );
            chdev_ptr = (chdev_t *)GET_PTR( chdev_xp );
            if( chdev_cxy == local_cxy )
            {
                vfs_add_child_in_parent( local_cxy,
                                         INODE_TYPE_DEV,
                                         FS_TYPE_DEVFS,
                                         devfs_external_inode_xp,
                                         chdev_ptr->name,
                                         GET_PTR( chdev_xp ),
                                         &inode_xp );
            }
        }
    }

    // create an IOC inode in each cluster containing an IOC chdev 
    for( channel = 0 ; channel < CONFIG_MAX_IOC_CHANNELS ; channel++ )
    {
        chdev_xp = chdev_dir.ioc[channel];
        if( chdev_xp != XPTR_NULL )
        {
            chdev_cxy = GET_CXY( chdev_xp );
            chdev_ptr = (chdev_t *)GET_PTR( chdev_xp );
            if( chdev_cxy == local_cxy )
            {
                vfs_add_child_in_parent( local_cxy,
                                         INODE_TYPE_DEV,
                                         FS_TYPE_DEVFS,
                                         devfs_external_inode_xp,
                                         chdev_ptr->name,
                                         GET_PTR( chdev_xp ),
                                         &inode_xp );
            }
        }
    }

    // create a FBF inode in each cluster containing a FBF chdev
    for( channel = 0 ; channel < CONFIG_MAX_IOC_CHANNELS ; channel++ )
    {
        chdev_xp = chdev_dir.fbf[channel];
        if( chdev_xp != XPTR_NULL )
        {
            chdev_cxy = GET_CXY( chdev_xp );
            chdev_ptr = (chdev_t *)GET_PTR( chdev_xp );
            if( chdev_cxy == local_cxy )
            {
                vfs_add_child_in_parent( local_cxy,
                                         INODE_TYPE_DEV,
                                         FS_TYPE_DEVFS,
                                         devfs_external_inode_xp,
                                         chdev_ptr->name,
                                         GET_PTR( chdev_xp ),
                                         &inode_xp );
            }
        }
    }

    // create a NIC_RX inode in each cluster containing a NIC_RX chdev 
    for( channel = 0 ; channel < CONFIG_MAX_NIC_CHANNELS ; channel++ )
    {
        chdev_xp = chdev_dir.nic_rx[channel];
        if( chdev_xp != XPTR_NULL )
        {
            chdev_cxy = GET_CXY( chdev_xp );
            chdev_ptr = (chdev_t *)GET_PTR( chdev_xp );
            if( chdev_cxy == local_cxy )
            {
                vfs_add_child_in_parent( local_cxy,
                                         INODE_TYPE_DEV,
                                         FS_TYPE_DEVFS,
                                         devfs_external_inode_xp,
                                         chdev_ptr->name,
                                         GET_PTR( chdev_xp ),
                                         &inode_xp );
            }
        }
    }

    // create a NIC_TX inode in each cluster containing a NIC_TX chdev 
    for( channel = 0 ; channel < CONFIG_MAX_NIC_CHANNELS ; channel++ )
    {
        chdev_xp = chdev_dir.nic_tx[channel];
        if( chdev_xp != XPTR_NULL )
        {
            chdev_cxy = GET_CXY( chdev_xp );
            chdev_ptr = (chdev_t *)GET_PTR( chdev_xp );
            if( chdev_cxy == local_cxy )
            {
                vfs_add_child_in_parent( local_cxy,
                                         INODE_TYPE_DEV,
                                         FS_TYPE_DEVFS,
                                         devfs_external_inode_xp,
                                         chdev_ptr->name,
                                         GET_PTR( chdev_xp ),
                                         &inode_xp );
            }
        }
    }

#if CONFIG_DEBUG_DEVFS_INIT
cycle = (uint32_t)hal_get_cycles();
if( CONFIG_DEBUG_DEVFS_INIT < cycle )
printk("\n[DBG] %s : thread %x exit at cycle %d\n",
__FUNCTION__ , CURRENT_THREAD , cycle );
#endif

}  // end devfs_local_init()

//////////////////////////////////////////
int devfs_user_move( bool_t     to_buffer,
                     xptr_t     file_xp,
                     void     * u_buf,
                     uint32_t   size )
{
    assert( ( file_xp != XPTR_NULL ) , __FUNCTION__ , "file_xp == XPTR_NULL" );

    assert( ( size < CONFIG_TXT_KBUF_SIZE ) , __FUNCTION__ , "string size too large" );

    xptr_t             chdev_xp;
    cxy_t              chdev_cxy;
    chdev_t          * chdev_ptr;    // associated chdev type
    uint32_t           func;         // chdev functionnal type
    uint32_t           channel;      // chdev channel index
    error_t            error;

    char               k_buf[CONFIG_TXT_KBUF_SIZE];  // local kernel buffer

#if (CONFIG_DEBUG_SYS_READ & 1)
enter_devfs_read = hal_time_stamp();
#endif

#if (CONFIG_DEBUG_SYS_WRITE & 1)
enter_devfs_write = hal_time_stamp();
#endif

#if CONFIG_DEBUG_DEVFS_MOVE
uint32_t cycle = (uint32_t)hal_get_cycles();
if( CONFIG_DEBUG_DEVFS_MOVE < cycle )
printk("\n[DBG] %s : thread %x enter / to_mem %d / cycle %d\n",
__FUNCTION__ , CURRENT_THREAD , to_buffer , cycle );
#endif

    // get extended pointer on chdev_xp
    chdev_xp = chdev_from_file( file_xp );

    // get cluster and local pointer on chdev
    chdev_cxy  = GET_CXY( chdev_xp );
    chdev_ptr  = (chdev_t *)GET_PTR( chdev_xp );

    // get chdev functionnal type and channel
    func    = hal_remote_lw( XPTR( chdev_cxy , &chdev_ptr->func ) );
    channel = hal_remote_lw( XPTR( chdev_cxy , &chdev_ptr->channel ) );

    // action depends on "func" and "to_buffer" 
    if( func == DEV_FUNC_TXT )
    {
        if( to_buffer )     // TXT read
        { 
            uint32_t i;
            for( i = 0 ; i < size ; i++ )
            {
                error = dev_txt_read( channel , &k_buf[i] );

                if( error ) 
                {
                    return -1;
                }
                else
                {
                    hal_strcpy_to_uspace( u_buf , k_buf , size );
                }
             }

#if CONFIG_DEBUG_DEVFS_MOVE
cycle = (uint32_t)hal_get_cycles();
if( CONFIG_DEBUG_DEVFS_MOVE < cycle )
printk("\n[DBG] %s : thread %x exit / to_mem %d / cycle %d\n",
__FUNCTION__ , CURRENT_THREAD , to_buffer / cycle );
#endif

#if (CONFIG_DEBUG_SYS_READ & 1)
exit_devfs_read = hal_time_stamp();
#endif
            return size;
        } 
        else                // TXT write  
        {
            hal_strcpy_from_uspace( k_buf , u_buf , size );

            error = dev_txt_write( channel , k_buf , size );
            if( error ) 
            {
                return -1;
            }
            else
            {

#if CONFIG_DEBUG_DEVFS_MOVE
cycle = (uint32_t)hal_get_cycles();
if( CONFIG_DEBUG_DEVFS_MOVE < cycle )
printk("\n[DBG] %s : thread %x exit / to_mem %d / cycle %d\n",
__FUNCTION__ , CURRENT_THREAD , to_buffer / cycle );
#endif

#if (CONFIG_DEBUG_SYS_WRITE & 1)
exit_devfs_write = hal_time_stamp();
#endif
                return size;
            }
        }
    }
    else
    {
        assert( false , __FUNCTION__ ,
        "%s does not support direct user access", chdev_func_str(func) );

        return -1;
    }

}  // end devfs_user_move()


