/*
 * rwlock.h - kernel read/write lock definition.
 * 
 * Author   Alain Greiner    (2016,2017,2018)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-kernel; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef _RWLOCK_H_
#define _RWLOCK_H_

#include <kernel_config.h>
#include <hal_types.h>
#include <list.h>

/*******************************************************************************************
 * This structure defines a local rwlock, that supports several simultaneous read 
 * accesses, but only one write access. It implements a ticket based allocation policy.
 * Both readers and writers must take a ticket before doing anything else, and access
 * are done in same order as requests (for both read an write ).
 * - A reader take the lock to atomically increments the registered readers count.
 *   Then it release the lock and access the protected structure. It atomically decrement
 *   the readers count without taking the lock when access is completed.
 * - A writer take the lock and keep it, but must wait completion of all current read 
 *   accesses before starting its own access.
 * As this local lock is only accessed by the local threads, if the lock is taken, 
 * the new-comers use a busy waiting policy with a delay between retry.
 * TODO : Introduce the rwlocks in the list of locks taken by a given thread for debug.
 ******************************************************************************************/

/****     Forward declarations    ****/

struct thread_s;

/*******************************************************************************************
 * This structure defines a local rwlock.
 * The "owner" and "list" fields are used for debug.
 ******************************************************************************************/

typedef struct rwlock_s
{
	uint32_t            ticket;           /*! first free ticket index                     */
    uint32_t            current;          /*! ticket index of current owner               */
    uint32_t            count;            /*! number of simultaneous readers threads      */

#if CONFIG_DEBUG_RWLOCKS
	struct thread_s   * owner;            /*! pointer on curent writer thread             */
    list_entry_t        list;             /*! member of list of locks taken by owner      */
#endif

}
rwlock_t;

/*******************************************************************************************
 * This function initializes a local rwlock.
 *******************************************************************************************
 * @ lock       : pointer on rwlock
 ******************************************************************************************/
void rwlock_init( rwlock_t * lock );

/*******************************************************************************************
 * This function get access to a local rwlock for a reader.
 *******************************************************************************************
 * @ lock       : pointer on rwlock
 ******************************************************************************************/
void rwlock_rd_lock( rwlock_t * lock );

/*******************************************************************************************
 * This function get access to a local rwlock for a writer.
 *******************************************************************************************
 * @ lock       : pointer on rwlock
 ******************************************************************************************/
void rwlock_wr_lock( rwlock_t * lock );

/*******************************************************************************************
 * This function unlocks a local rwlock for a reader.
 *******************************************************************************************
 * @ lock       : pointer on rwlock
 ******************************************************************************************/
void rwlock_rd_unlock( rwlock_t * lock );

/*******************************************************************************************
 * This function unlocks a local rwlock for a writer.
 *******************************************************************************************
 * @ lock       : pointer on rwlock
 ******************************************************************************************/
void rwlock_wr_unlock( rwlock_t * lock );


#endif	/* _RWLOCK_H_ */
