/*
 * scheduler.c - Core scheduler implementation.
 * 
 * Author    Alain Greiner (2016)
 *
 * Copyright (c)  UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH. is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH. is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH.; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <kernel_config.h>
#include <hal_types.h>
#include <hal_switch.h>
#include <hal_irqmask.h>
#include <hal_context.h>
#include <printk.h>
#include <list.h>
#include <core.h>
#include <thread.h>
#include <chdev.h>
#include <scheduler.h>

///////////////////////////////////////////////////////////////////////////////////////////
// Extern global variables
///////////////////////////////////////////////////////////////////////////////////////////

extern chdev_directory_t    chdev_dir;            // allocated in kernel_init.c file
extern uint32_t             switch_save_sr[];     // allocated in kernel_init.c file

////////////////////////////////
void sched_init( core_t * core )
{
    scheduler_t * sched = &core->scheduler;

    sched->u_threads_nr   = 0;
    sched->k_threads_nr   = 0;

    sched->current        = CURRENT_THREAD;
    sched->idle           = NULL;             // initialized in kernel_init()
    sched->u_last         = NULL;             // initialized in sched_register_thread()
    sched->k_last         = NULL;             // initialized in sched_register_thread()

    // initialise threads lists
    list_root_init( &sched->u_root );
    list_root_init( &sched->k_root );

    sched->req_ack_pending = false;           // no pending request

}  // end sched_init()

////////////////////////////////////////////
void sched_register_thread( core_t   * core,
                            thread_t * thread )
{
    scheduler_t * sched = &core->scheduler;
    thread_type_t type  = thread->type;

    // take lock protecting sheduler lists
    spinlock_lock( &sched->lock );

    if( type == THREAD_USER )
    {
        list_add_last( &sched->u_root , &thread->sched_list );
        sched->u_threads_nr++;
        if( sched->u_last == NULL ) sched->u_last = &thread->sched_list;
    }
    else // kernel thread
    {
        list_add_last( &sched->k_root , &thread->sched_list );
        sched->k_threads_nr++;
        if( sched->k_last == NULL ) sched->k_last = &thread->sched_list; 
    }

    // release lock 
    hal_fence();
    spinlock_unlock( &sched->lock );

}  // end sched_register_thread()

//////////////////////////////////////////////
thread_t * sched_select( scheduler_t * sched )
{
    thread_t     * thread;
    list_entry_t * current;
    list_entry_t * last;
    list_entry_t * root;
    bool_t         done;

    // take lock protecting sheduler lists
    spinlock_lock( &sched->lock );

    // first : scan the kernel threads list if not empty 
    if( list_is_empty( &sched->k_root ) == false )
    {
        root    = &sched->k_root;
        last    = sched->k_last;
        current = last;
        done    = false;

        while( done == false )
        {
            // get next entry in kernel list
            current = current->next;

            // check exit condition
            if( current == last ) done = true;

            // skip the root that does not contain a thread
            if( current == root ) continue;

            // get thread pointer for this entry
            thread = LIST_ELEMENT( current , thread_t , sched_list );

            // analyse kernel thread type
            switch( thread->type )
            {
                case THREAD_RPC:  // if non blocked and RPC FIFO non-empty
                if( (thread->blocked == 0) && 
                    (local_fifo_is_empty( &LOCAL_CLUSTER->rpc_fifo ) == 0) )
                {
                    spinlock_unlock( &sched->lock );
                    return thread;
                }
                break;

                case THREAD_DEV:  // if non blocked and waiting queue non empty
                if( (thread->blocked == 0) &&
                    (xlist_is_empty( XPTR( local_cxy , &thread->chdev->wait_root)) == 0) ) 
                {
                    spinlock_unlock( &sched->lock );
                    return thread;
                }
                break;

                default:
                break;
            }
        } // end loop on kernel threads
    } // end if kernel threads

    // second : scan the user threads list if not empty 
    if( list_is_empty( &sched->u_root ) == false )
    {
        root    = &sched->u_root;
        last    = sched->u_last;
        current = last;
        done    = false;

        while( done == false )
        {
            // get next entry in user list
            current = current->next;

            // check exit condition
            if( current == last ) done = true;

            // skip the root that does not contain a thread
            if( current == root ) continue;

            // get thread pointer for this entry
            thread = LIST_ELEMENT( current , thread_t , sched_list );

            // return thread if runnable
            if( thread->blocked == 0 )
            {
                spinlock_unlock( &sched->lock );
                return thread;
            }
        } // end loop on user threads
    } // end if user threads

    // third : return idle thread if no other runnable thread
    spinlock_unlock( &sched->lock );
    return sched->idle;

}  // end sched_select()

///////////////////////////////////////////
void sched_handle_signals( core_t * core )
{

    list_entry_t * iter;
    thread_t     * thread;
    process_t    * process;

    scheduler_t  * sched = &core->scheduler;

    // take lock protecting threads lists
    spinlock_lock( &sched->lock );

    // scan all user threads
    LIST_FOREACH( &sched->u_root , iter )
    {
        thread = LIST_ELEMENT( iter , thread_t , sched_list );

        // handle REQ_ACK 
        if( thread->flags & THREAD_FLAG_REQ_ACK )
        {
            // check thread blocked
            assert( (thread->blocked & THREAD_BLOCKED_GLOBAL) , 
            __FUNCTION__ , "thread not blocked" );
 
            // decrement response counter
            hal_atomic_add( thread->ack_rsp_count , -1 );

            // reset REQ_ACK in thread descriptor
            thread_reset_req_ack( thread );
        }

        // handle REQ_DELETE
        if( thread->flags & THREAD_FLAG_REQ_DELETE )
        {
            // get thread process descriptor
            process = thread->process;

#if CONFIG_DEBUG_SCHED_HANDLE_SIGNALS
uint32_t cycle = (uint32_t)hal_get_cycles();
if( CONFIG_DEBUG_SCHED_HANDLE_SIGNALS < cycle )
printk("\n[DBG] %s : thread %x in proces %x must be deleted / cycle %d\n",
__FUNCTION__ , thread , process->pid , cycle );
#endif
	        // release FPU if required
	        if( thread->core->fpu_owner == thread )  thread->core->fpu_owner = NULL;

            // detach thread from parent if attached
            if( (thread->flags & THREAD_FLAG_DETACHED) == 0 ) 
            thread_child_parent_unlink( thread->parent , XPTR( local_cxy , thread ) );

            // remove thread from scheduler (scheduler lock already taken)
            uint32_t threads_nr = sched->u_threads_nr;
            assert( (threads_nr != 0) , __FUNCTION__ , "u_threads_nr cannot be 0\n" );
            sched->u_threads_nr = threads_nr - 1;
            list_unlink( &thread->sched_list );
            if( threads_nr == 1 ) sched->u_last = NULL;

            // delete thread
            thread_destroy( thread );

#if CONFIG_DEBUG_SCHED_HANDLE_SIGNALS
cycle = (uint32_t)hal_get_cycles();
if( CONFIG_DEBUG_SCHED_HANDLE_SIGNALS < cycle )
printk("\n[DBG] %s : thread %x in process %x has been deleted / cycle %d\n",
__FUNCTION__ , thread , process->pid , cycle );
#endif
            // destroy process descriptor if no more threads
            if( process->th_nr == 0 ) 
            {
                // delete process    
                process_destroy( process );

#if CONFIG_DEBUG_SCHED_HANDLE_SIGNALS
cycle = (uint32_t)hal_get_cycles();
if( CONFIG_DEBUG_SCHED_HANDLE_SIGNALS < cycle )
printk("\n[DBG] %s : process %x has been deleted / cycle %d\n",
__FUNCTION__ , process->pid , cycle );
#endif

            }
        }
    }

    // release lock 
    hal_fence();
    spinlock_unlock( &sched->lock );

} // end sched_handle_signals()

////////////////////////////////
void sched_yield( char * cause )
{
    thread_t    * next;
    thread_t    * current = CURRENT_THREAD;
    core_t      * core    = current->core;
    scheduler_t * sched   = &core->scheduler;
 
#if (CONFIG_DEBUG_SCHED_YIELD & 0x1)
if( CONFIG_DEBUG_SCHED_YIELD < (uint32_t)hal_get_cycles() )
sched_display( core->lid );
#endif

    // delay the yield if current thread has locks
    if( (current->local_locks != 0) || (current->remote_locks != 0) )
    {
        current->flags |= THREAD_FLAG_SCHED;
        return;
    }

    // enter critical section / save SR in current thread descriptor
    hal_disable_irq( &CURRENT_THREAD->save_sr );

    // loop on threads to select next thread 
    next = sched_select( sched );

    // check next thread kernel_stack overflow
    assert( (next->signature == THREAD_SIGNATURE),
    __FUNCTION__ , "kernel stack overflow for thread %x\n", next );

    // check next thread attached to same core as the calling thread
    assert( (next->core == current->core),
    __FUNCTION__ , "next core %x != current core %x\n", next->core, current->core );

    // check next thread not blocked when type != IDLE
    assert( ((next->blocked == 0) || (next->type == THREAD_IDLE)) , __FUNCTION__ ,
    "next thread %x (%s) is blocked on core[%x,%d]\n", 
    next->trdid , thread_type_str(next->type) , local_cxy , core->lid );

    // switch contexts and update scheduler state if next != current
	if( next != current )
    {

#if CONFIG_DEBUG_SCHED_YIELD
uint32_t cycle = (uint32_t)hal_get_cycles();
if( CONFIG_DEBUG_SCHED_YIELD < cycle )
printk("\n[DBG] %s : core[%x,%d] / cause = %s\n"
"      thread %x (%s) (%x,%x) => thread %x (%s) (%x,%x) / cycle %d\n",
__FUNCTION__, local_cxy, core->lid, cause, 
current, thread_type_str(current->type), current->process->pid, current->trdid,
next , thread_type_str(next->type) , next->process->pid , next->trdid , cycle );
#endif

        // update scheduler 
        sched->current = next;
        if( next->type == THREAD_USER ) sched->u_last = &next->sched_list;
        else                            sched->k_last = &next->sched_list;

        // handle FPU ownership
	    if( next->type == THREAD_USER )
        {
	        if( next == current->core->fpu_owner )  hal_fpu_enable();
	        else                                    hal_fpu_disable();
        }

        // switch CPU from current thread context to new thread context
        hal_do_cpu_switch( current->cpu_context, next->cpu_context );
    }
    else
    {

#if (CONFIG_DEBUG_SCHED_YIELD & 1)
uint32_t cycle = (uint32_t)hal_get_cycles();
if( CONFIG_DEBUG_SCHED_YIELD < cycle )
printk("\n[DBG] %s : core[%x,%d] / cause = %s\n"
"      thread %x (%s) (%x,%x) continue / cycle %d\n",
__FUNCTION__, local_cxy, core->lid, cause,
current, thread_type_str(current->type), current->process->pid, current->trdid, cycle );
#endif

    }

    // handle pending requests for all threads executing on this core.
    sched_handle_signals( core );

    // exit critical section / restore SR from current thread descriptor
    hal_restore_irq( CURRENT_THREAD->save_sr );

}  // end sched_yield()


///////////////////////////////
void sched_display( lid_t lid )
{
    list_entry_t * iter;
    thread_t     * thread;
    uint32_t       save_sr;

    assert( (lid < LOCAL_CLUSTER->cores_nr), __FUNCTION__, "illegal core index %d\n", lid);

    core_t       * core    = &LOCAL_CLUSTER->core_tbl[lid];
    scheduler_t  * sched   = &core->scheduler;
    
    // get pointers on TXT0 chdev
    xptr_t    txt0_xp  = chdev_dir.txt_tx[0];
    cxy_t     txt0_cxy = GET_CXY( txt0_xp );
    chdev_t * txt0_ptr = GET_PTR( txt0_xp );

    // get extended pointer on remote TXT0 chdev lock
    xptr_t  lock_xp = XPTR( txt0_cxy , &txt0_ptr->wait_lock );

    // get TXT0 lock in busy waiting mode
    remote_spinlock_lock_busy( lock_xp , &save_sr );

    nolock_printk("\n***** threads on core[%x,%d] / current %x / cycle %d\n",
            local_cxy , core->lid, sched->current, (uint32_t)hal_get_cycles() );

    // display kernel threads
    LIST_FOREACH( &sched->k_root , iter )
    {
        thread = LIST_ELEMENT( iter , thread_t , sched_list );
        if (thread->type == THREAD_DEV) 
        {
            nolock_printk(" - %s / pid %X / trdid %X / desc %X / block %X / flags %X / %s\n",
            thread_type_str( thread->type ), thread->process->pid, thread->trdid,
            thread, thread->blocked, thread->flags, thread->chdev->name );
        }
        else
        {
            nolock_printk(" - %s / pid %X / trdid %X / desc %X / block %X / flags %X\n",
            thread_type_str( thread->type ), thread->process->pid, thread->trdid,
            thread, thread->blocked, thread->flags );
        }
    }

    // display user threads
    LIST_FOREACH( &sched->u_root , iter )
    {
        thread = LIST_ELEMENT( iter , thread_t , sched_list );
        nolock_printk(" - %s / pid %X / trdid %X / desc %X / block %X / flags %X\n",
        thread_type_str( thread->type ), thread->process->pid, thread->trdid,
        thread, thread->blocked, thread->flags );
    }

    // release TXT0 lock
    remote_spinlock_unlock_busy( lock_xp , save_sr );

}  // end sched_display()

