/*
 * fatfs.c - FATFS file system API implementation.
 *
 * Author    Alain Greiner (2016,2017)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */


#include <hal_types.h>
#include <hal_special.h>
#include <printk.h>
#include <thread.h>
#include <kmem.h>
#include <ppm.h>
#include <vfs.h>
#include <string.h>
#include <rpc.h>
#include <mapper.h>
#include <cluster.h>
#include <dev_ioc.h>
#include <fatfs.h>


//////////////////////////////////////////////////////////////////////////////////////////
//          Extern  variables         
//////////////////////////////////////////////////////////////////////////////////////////

extern vfs_ctx_t          fs_context[FS_TYPES_NR];   // allocated in vfs.c file

extern remote_barrier_t   global_barrier;            // allocated in kernel_init.c
 
//////////////////////////////////////////////////////////////////////////////////////////
//              FATFS specific and static functions 
//////////////////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////////////////
// These functions return the "offset" and "length" values of an 
// [offset,length] constant defined in the fatfs.h file.
//////////////////////////////////////////////////////////////////////////////////////////

static inline int get_length( int offset , int length ) { return length; }

static inline int get_offset( int offset , int length ) { return offset; }


//////////////////////////////////////////////////////////////////////////////////////////
// This static function returns the LBA of the first sector of a FAT cluster.
// This function can be called by any thread running in any cluster.
//////////////////////////////////////////////////////////////////////////////////////////
// @ ctx	  :	pointer on FATFS context.
// @ cluster  :	cluster index in FATFS.
// @ return the lba value.
//////////////////////////////////////////////////////////////////////////////////////////
static inline uint32_t fatfs_lba_from_cluster( fatfs_ctx_t * ctx,
                                               uint32_t      cluster )
{
    return (ctx->cluster_begin_lba + ((cluster - 2) << 3));
}

//////////////////////////////////////////////////////////////////////////////////////////
// This function return an integer record value (one, two, or four bytes) 
// from a memory buffer, taking into account endianness.
//////////////////////////////////////////////////////////////////////////////////////////
// @ offset        : first byte of record in buffer.
// @ size          : record length in bytes (1/2/4).
// @ buffer        : pointer on buffer base.
// @ little endian : the most significant byte has the highest address when true.
// @ return the integer value in a 32 bits word.
//////////////////////////////////////////////////////////////////////////////////////////
static uint32_t fatfs_get_record( uint32_t    offset,
                                  uint32_t    size,
                                  uint8_t   * buffer,
                                  uint32_t    little_endian )
{
    uint32_t n;
    uint32_t res  = 0;

    if ( little_endian)
    {
        for( n = size ; n > 0 ; n-- ) res = (res<<8) | buffer[offset+n-1];
    }
    else
    {
        for( n = 0 ; n < size ; n++ ) res = (res<<8) | buffer[offset+n];
    }
    return res;

}  // end fatfs_get_record()

//////////////////////////////////////////////////////////////////////////////////////////
// This static function retun in the <name> buffer a short name stored in
// a SFN FATFS directory entry.
/////////////////////////i////////////////////////////////////////////////////////////////
// @ buffer   : pointer on buffer containing the directory entry.
// @ name     : [out] buffer allocated by the caller.
//////////////////////////////////////////////////////////////////////////////////////////
static void fatfs_get_name_from_short( uint8_t * buffer,
                                       char    * name )
{
    uint32_t i;
    uint32_t j = 0;

    // get name
    for ( i = 0; i < 8 && buffer[i] != ' '; i++ )
    {
        name[j] = to_lower( buffer[i] );
        j++;
    }

    // get extension
    for ( i = 8; i < 8 + 3 && buffer[i] != ' '; i++ )
    {
        // we entered the loop so there is an extension. add the dot
        if ( i == 8 )
        {
            name[j] = '.';
            j++;
        }

        name[j] = to_lower( buffer[i] );
        j++;
    }

    name[j] = '\0';
}

//////////////////////////////////////////////////////////////////////////////////////////
// This static function retun in the <name> buffer a partial name stored in
// a LFN FATFS directory entry.
/////////////////////////i////////////////////////////////////////////////////////////////
// @ buffer   : pointer on buffer containing the directory entry.
// @ name     : [out] buffer allocated by the caller.
//////////////////////////////////////////////////////////////////////////////////////////
static void fatfs_get_name_from_long( uint8_t * buffer,
                                      char    * name )
{
    uint32_t   name_offset   = 0;
    uint32_t   buffer_offset = get_length(LDIR_ORD);
    uint32_t   l_name_1      = get_length(LDIR_NAME_1);
    uint32_t   l_name_2      = get_length(LDIR_NAME_2);
    uint32_t   l_name_3      = get_length(LDIR_NAME_3);
    uint32_t   l_attr        = get_length(LDIR_ATTR);
    uint32_t   l_type        = get_length(LDIR_TYPE);
    uint32_t   l_chksum      = get_length(LDIR_CHKSUM);
    uint32_t   l_rsvd        = get_length(LDIR_RSVD);

    uint32_t   j             = 0;
    uint32_t   eof           = 0;

    while ( (buffer_offset != DIR_ENTRY_SIZE)  && (!eof) )
    {
        while (j != l_name_1 && !eof )
        {
            if ( (buffer[buffer_offset] == 0x00) || 
                 (buffer[buffer_offset] == 0xFF) )
            {
                eof = 1;
                continue;
            }
            name[name_offset] = buffer[buffer_offset];
            buffer_offset += 2;
            j += 2;
            name_offset++;
        }

        buffer_offset += (l_attr + l_type + l_chksum);
        j = 0;

        while (j != l_name_2 && !eof )
        {
            if ( (buffer[buffer_offset] == 0x00) || 
                 (buffer[buffer_offset] == 0xFF) )
            {
                eof = 1;
                continue;
            }
            name[name_offset] = buffer[buffer_offset];
            buffer_offset += 2;
            j += 2;
            name_offset++;
        }

        buffer_offset += l_rsvd;
        j = 0;

        while (j != l_name_3 && !eof )
        {
            if ( (buffer[buffer_offset] == 0x00) || 
                 (buffer[buffer_offset] == 0xFF) )
            {
                eof = 1;
                continue;
            }
            name[name_offset] = buffer[buffer_offset];
            buffer_offset += 2;
            j += 2;
            name_offset++;
        }
    }
    name[name_offset] = 0;

} // end get_name_from_long()


//////////////////////////////////////////////////////////////////////////////////////////
//              FATFS specific but extern functions 
//////////////////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////////////////
void fatfs_ctx_display()
{
    vfs_ctx_t   * vfs_ctx   = &fs_context[FS_TYPE_FATFS];
    fatfs_ctx_t * fatfs_ctx = (fatfs_ctx_t *)vfs_ctx->extend;

    printk("\n*** FAT context ***\n" 
           "- fat_sectors      = %d\n"
           "- sector size      = %d\n"
           "- cluster size     = %d\n"
           "- fat_first_lba    = %d\n"
           "- data_first_lba   = %d\n"
           "- root_dir_cluster = %d\n"
           "- mapper_xp        = %l\n",
           fatfs_ctx->fat_sectors_count,
           fatfs_ctx->bytes_per_sector,
           fatfs_ctx->sectors_per_cluster * fatfs_ctx->bytes_per_sector,
           fatfs_ctx->fat_begin_lba,
           fatfs_ctx->cluster_begin_lba,
           fatfs_ctx->root_dir_cluster,
           fatfs_ctx->fat_mapper_xp );
}

/////////////////////////////////////////////
error_t fatfs_get_cluster( mapper_t * mapper,
                           uint32_t   first_cluster_id,
                           uint32_t   searched_page_index,
                           uint32_t * searched_cluster_id )
{
    page_t   * current_page_desc;      // pointer on current page descriptor
    uint32_t * current_page_buffer;    // pointer on current page (array of uint32_t)
    uint32_t   current_page_index;     // index of current page in FAT 
    uint32_t   current_page_offset;    // offset of slot in current page
    uint32_t   page_count_in_file;     // index of page in file (index in linked list)
    uint32_t   next_cluster_id;        // content of current FAT slot

    assert( (searched_page_index > 0) , __FUNCTION__ , 
    "no FAT access required for first page\n");

#if DEBUG_FATFS_GET_CLUSTER
uint32_t cycle = (uint32_t)hal_get_cycles();
if( DEBUG_FATFS_GET_CLUSTER < cycle )
printk("\n[DBG] %s : thread %x enter / first_cluster_id %d / searched_index / cycle %d\n",
__FUNCTION__, CURRENT_THREAD, first_cluster_id, searched_page_index, cycle );
#endif

    // get number of FAT slots per page
    uint32_t slots_per_page = CONFIG_PPM_PAGE_SIZE >> 2;

    // initialize loop variable
    current_page_index  = first_cluster_id / slots_per_page;
    current_page_offset = first_cluster_id % slots_per_page;
    page_count_in_file  = 0;
    next_cluster_id     = 0xFFFFFFFF;

    // scan FAT (i.e. traverse FAT linked list)
    while( page_count_in_file < searched_page_index )
    {
        // get pointer on current page descriptor
        current_page_desc = mapper_get_page( mapper , current_page_index );

        if( current_page_desc == NULL ) return EIO;

        // get pointer on buffer for current page
        xptr_t base_xp = ppm_page2base( XPTR( local_cxy , current_page_desc ) );
        current_page_buffer = (uint32_t *)GET_PTR( base_xp );

        // get FAT slot content 
        next_cluster_id = current_page_buffer[current_page_offset];

#if (DEBUG_FATFS_GET_CLUSTER & 1)
if( DEBUG_FATFS_GET_CLUSTER < cycle )
printk("\n[DBG] %s : traverse FAT / current_page_index = %d\n"
"current_page_offset = %d / next_cluster_id = %d\n",
__FUNCTION__, current_page_index, current_page_offset , next_cluster_id );
#endif

        // update loop variables
        current_page_index  = next_cluster_id / slots_per_page;
        current_page_offset = next_cluster_id % slots_per_page;
        page_count_in_file++;
    }

    if( next_cluster_id == 0xFFFFFFFF ) return EIO;
   
#if DEBUG_FATFS_GET_CLUSTER
cycle = (uint32_t)hal_get_cycles();
if( DEBUG_FATFS_GET_CLUSTER < cycle )
printk("\n[DBG] %s : thread %x exit / searched_cluster_id = %d / cycle %d\n",
__FUNCTION__, CURRENT_THREAD, next_cluster_id / cycle );
#endif

    *searched_cluster_id = next_cluster_id;
    return 0;

}  // end fatfs_get_cluster()



///////////////////////////////////////////////////////////////////////////////////////
// Generic API : the following functions are called by the kernel (VFS)
//               and must be defined by all supported file systems.
///////////////////////////////////////////////////////////////////////////////////////

///////////////////////////////
fatfs_ctx_t * fatfs_ctx_alloc()
{
    kmem_req_t    req;
	req.type    = KMEM_FATFS_CTX;
	req.size    = sizeof(fatfs_ctx_t);
    req.flags   = AF_KERNEL | AF_ZERO;

	return (fatfs_ctx_t *)kmem_alloc( &req );
}

//////////////////////////////////////////////
void fatfs_ctx_init( fatfs_ctx_t * fatfs_ctx )
{
    error_t       error;
    kmem_req_t    req;
    uint8_t     * buffer;

#if DEBUG_FATFS_INIT
uint32_t cycle = (uint32_t)hal_get_cycles();
if( DEBUG_FATFS_INIT < cycle )
printk("\n[DBG] %s : thread %x enter for fatfs_ctx = %x / cycle %d\n",
__FUNCTION__ , CURRENT_THREAD , fatfs_ctx , cycle );
#endif

    assert( (fatfs_ctx != NULL) , __FUNCTION__ ,
    "cannot allocate memory for FATFS context\n" );

    // allocate a 512 bytes buffer to store the boot record
	req.type    = KMEM_512_BYTES;
    req.flags   = AF_KERNEL | AF_ZERO;
	buffer      = (uint8_t *)kmem_alloc( &req );

    assert( (buffer != NULL) , __FUNCTION__ ,
    "cannot allocate memory for 512 bytes buffer\n" );
     
    // load the boot record from device
    // using a synchronous access to IOC device  
    error = dev_ioc_sync_read( buffer , 0 , 1 );

    assert( (error == 0) , __FUNCTION__ , 
    "cannot access boot record\n" );

#if (DEBUG_FATFS_INIT & 0x1)
if( DEBUG_FATFS_INIT < cycle )
{
    uint32_t   line;
    uint32_t   byte = 0;
    printk("\n***** %s : FAT boot record\n", __FUNCTION__ );
    for ( line = 0 ; line < 32 ; line++ )
    {
        printk(" %X | %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x |\n",
               byte,
               buffer[byte+ 0],buffer[byte+ 1],buffer[byte+ 2],buffer[byte+ 3],
               buffer[byte+ 4],buffer[byte+ 5],buffer[byte+ 6],buffer[byte+ 7],
               buffer[byte+ 8],buffer[byte+ 9],buffer[byte+10],buffer[byte+11],
               buffer[byte+12],buffer[byte+13],buffer[byte+14],buffer[byte+15] );

         byte += 16;
    }
}
#endif

    // check sector size from boot record
    uint32_t sector_size = fatfs_get_record( BPB_BYTSPERSEC , buffer , 1 );

    assert( (sector_size == 512) , __FUNCTION__ ,
            "sector size must be 512 bytes\n" );

    // check cluster size from boot record
    uint32_t nb_sectors = fatfs_get_record( BPB_SECPERCLUS , buffer , 1 );

    assert( (nb_sectors == 8) , __FUNCTION__ ,
    "cluster size must be 8 sectors\n" );

    // check number of FAT copies from boot record
    uint32_t nb_fats = fatfs_get_record( BPB_NUMFATS , buffer , 1 );

    assert( (nb_fats == 1) , __FUNCTION__ ,
    "number of FAT copies must be 1\n" );

    // get & check number of sectors in FAT from boot record
    uint32_t fat_sectors = fatfs_get_record( BPB_FAT32_FATSZ32 , buffer , 1 );

    assert( ((fat_sectors & 0xF) == 0) , __FUNCTION__ ,
    "FAT not multiple of 16 sectors\n");

    // get and check root cluster from boot record
    uint32_t root_cluster = fatfs_get_record( BPB_FAT32_ROOTCLUS , buffer , 1 );

    assert( (root_cluster == 2) , __FUNCTION__ ,
    "root cluster index must be  2\n");

    // get FAT lba from boot record
    uint32_t fat_lba = fatfs_get_record( BPB_RSVDSECCNT , buffer , 1 );

    // release the 512 bytes buffer
    req.type = KMEM_512_BYTES;
    req.ptr  = buffer;
    kmem_free( &req );

    // allocate a mapper for the FAT itself
    mapper_t * fat_mapper = mapper_create( FS_TYPE_FATFS );

    assert( (fat_mapper != NULL) , __FUNCTION__ ,
    "no memory for FAT mapper" );

    // WARNING : the inode field MUST be NULL for the FAT mapper
    fat_mapper->inode = NULL;

    // initialize the FATFS context
    fatfs_ctx->fat_begin_lba         = fat_lba;
    fatfs_ctx->fat_sectors_count     = fat_sectors; 
    fatfs_ctx->bytes_per_sector      = sector_size;
    fatfs_ctx->sectors_per_cluster   = nb_sectors;
    fatfs_ctx->cluster_begin_lba     = fat_lba + fat_sectors;
    fatfs_ctx->root_dir_cluster      = 2;
    fatfs_ctx->last_allocated_sector = 0;    // TODO ???
    fatfs_ctx->last_allocated_index  = 0;    // TODO ???
    fatfs_ctx->fat_mapper_xp         = XPTR( local_cxy , fat_mapper );

#if DEBUG_FATFS_INIT
cycle = (uint32_t)hal_get_cycles();
if( DEBUG_FATFS_INIT < cycle )
printk("\n[DBG] %s : thread %x exit for fatfs_ctx = %x / cycle %d\n",
__FUNCTION__ , CURRENT_THREAD , fatfs_ctx , cycle );
#endif

}  // end fatfs_ctx_init()

/////////////////////////////////////////////////
void fatfs_ctx_destroy( fatfs_ctx_t * fatfs_ctx )
{
    kmem_req_t    req;
    req.type = KMEM_FATFS_CTX;
    req.ptr  = fatfs_ctx;
    kmem_free( &req );
}

//////////////////////////////////////////////
error_t fatfs_mapper_move_page( page_t * page,
                                bool_t   to_mapper )
{
    error_t       error;
    vfs_inode_t * inode;
    mapper_t    * mapper;
    uint32_t      index;       // page index in mapper
    uint8_t     * buffer;      // page base address in mapper
    uint32_t      count;       // number of sectors in a page
    uint32_t      lba;         // block address on device
    fatfs_ctx_t * fatfs_ctx;   // pointer on local FATFS context

    // get pointer on mapper and page index from page descriptor
    mapper = page->mapper;
    index  = page->index;

    // get inode pointer from mapper
    inode = mapper->inode;

#if DEBUG_FATFS_MOVE
uint32_t cycle = (uint32_t)hal_get_cycles();
if( DEBUG_FATFS_MOVE < cycle )
printk("\n[DBG] %s : thread %x enter / page %d / inode %x / mapper %x / cycle %d\n",
__FUNCTION__ , CURRENT_THREAD , index , inode , mapper , cycle );
#endif

    // get page base address
    xptr_t base_xp = ppm_page2base( XPTR( local_cxy , page ) );
    buffer = (uint8_t *)GET_PTR( base_xp );
 
    // get number of sectors for one page (from FATFS context)
    fatfs_ctx = (fatfs_ctx_t *)fs_context[FS_TYPE_FATFS].extend;
    count = fatfs_ctx->sectors_per_cluster;

    // test FAT/normal inode
    if( inode == NULL )      // it is the FAT mapper
    {
        // get lba from page index 
        lba = fatfs_ctx->fat_begin_lba + (count * index);
 
#if (DEBUG_FATFS_MOVE & 0x1)
if( DEBUG_FATFS_MOVE < cycle )
printk("\n[DBG] %s : access FAT on device / lba = %d\n", __FUNCTION__ , lba );
#endif

        // access device
        if( to_mapper ) error = dev_ioc_sync_read ( buffer , lba , count );
        else            error = dev_ioc_write( buffer , lba , count );     

        if( error ) return EIO;
    }
    else                     // it is a normal inode mapper
    {
        uint32_t  searched_cluster_id;

        // get first_cluster_id from inode extension
        uint32_t  first_cluster_id = (uint32_t)(intptr_t)inode->extend;

        // compute cluster_id
        if( index == 0 )            // no need to access FAT mapper
        {
            searched_cluster_id = first_cluster_id;
        }
        else                        // FAT mapper access required
        {
            // get cluster and local pointer on FAT mapper
            xptr_t     fat_mapper_xp  = fatfs_ctx->fat_mapper_xp;
            cxy_t      fat_mapper_cxy = GET_CXY( fat_mapper_xp );
            mapper_t * fat_mapper_ptr = (mapper_t *)GET_PTR( fat_mapper_xp );

            // access FAT mapper
            if( fat_mapper_cxy == local_cxy )    // FAT mapper is local
            {

#if (DEBUG_FATFS_MOVE & 0x1)
if( DEBUG_FATFS_MOVE < cycle )
print("\n[DBG] %s : access local FAT mapper\n"
"fat_mapper_cxy = %x / fat_mapper_ptr = %x / first_cluster_id = %d / index = %d\n",
__FUNCTION__ , fat_mapper_cxy , fat_mapper_ptr , first_cluster_id , index );
#endif
                error = fatfs_get_cluster( fat_mapper_ptr,
                                           first_cluster_id,
                                           index,
                                           &searched_cluster_id );
            }
            else                                 // FAT mapper is remote
            {

#if (DEBUG_FATFS_MOVE & 0x1)
if( DEBUG_FATFS_MOVE < cycle )
printk("\n[DBG] %s : access remote FAT mapper\n"
"fat_mapper_cxy = %x / fat_mapper_ptr = %x / first_cluster_id = %d / index = %d\n",
__FUNCTION__ , fat_mapper_cxy , fat_mapper_ptr , first_cluster_id , index );
#endif
                rpc_fatfs_get_cluster_client( fat_mapper_cxy,
                                              fat_mapper_ptr,
                                              first_cluster_id,
                                              index,
                                              &searched_cluster_id,
                                              &error );
            }

            if( error )  return EIO;
        }

#if (DEBUG_FATFS_MOVE & 0x1)
if( DEBUG_FATFS_MOVE < cycle )
printk("\n[DBG] %s : access device for inode %x / cluster_id %d\n",
__FUNCTION__ , inode , searched_cluster_id );
#endif

        // get lba from cluster_id
        lba = fatfs_lba_from_cluster( fatfs_ctx , searched_cluster_id );

        // access device
        if( to_mapper ) error = dev_ioc_sync_read ( buffer , lba , count );
        else            error = dev_ioc_write( buffer , lba , count );     

        if( error ) return EIO;
    }

#if DEBUG_FATFS_MOVE
cycle = (uint32_t)hal_get_cycles();
if( DEBUG_FATFS_MOVE < cycle )
printk("\n[DBG] %s : thread %x exit / page %d / inode %x / mapper %x / cycle %d\n",
__FUNCTION__ , CURRENT_THREAD , index , inode , mapper , cycle );
#endif

#if (DEBUG_FATFS_MOVE & 0x1)
if( DEBUG_FATFS_MOVE < cycle )
{
    uint32_t * tab = (uint32_t *)buffer;
    uint32_t line , word;
    printk("\n***** %s : First 64 words of loaded page\n", __FUNCTION__ );
    for( line = 0 ; line < 8 ; line++ )
    {
        printk("%X : ", line );
        for( word = 0 ; word < 8 ; word++ ) printk("%X ", tab[(line<<3) + word] );
        printk("\n");
    }
}
#endif

    return 0;

}  // end fatfs_mapper_move_page()

/////////////////////////////////////////////////////
error_t fatfs_inode_load( vfs_inode_t * parent_inode,
                          char        * name,
                          xptr_t        child_inode_xp )
{
    // Two embedded loops:
    // - scan the parent mapper pages 
    // - scan the directory entries in each 4 Kbytes page

#if DEBUG_FATFS_LOAD
uint32_t cycle = (uint32_t)hal_get_cycles();
if( DEBUG_FATFS_LOAD < cycle )
printk("\n[DBG] %s : thread %x enter for child <%s> in parent inode %x / cycle %d\n",
__FUNCTION__ , CURRENT_THREAD , name , parent_inode , cycle );
#endif

    mapper_t * mapper = parent_inode->mapper;

    assert( (mapper != NULL) , __FUNCTION__ , "parent mapper undefined\n");
    
    char       cname[CONFIG_VFS_MAX_NAME_LENGTH];  // name extracter from each directory entry

    char       lfn1[16];         // buffer for one partial cname
    char       lfn2[16];         // buffer for one partial cname
    char       lfn3[16];         // buffer for one partial cname
    page_t   * page;             // pointer on current page descriptor
    uint8_t  * base;             // pointer on current page base
    uint32_t   offset  = 0;      // byte offset in page
    uint32_t   index   = 0;      // page index in mapper
    uint32_t   attr;             // directory entry ATTR field
    uint32_t   ord;              // directory entry ORD field
    uint32_t   seq;              // sequence index
    uint32_t   lfn     = 0;      // LFN entries number
    uint32_t   size    = 0;      // searched file/dir size (bytes) 
    uint32_t   cluster = 0;      // searched file/dir cluster index
    uint32_t   is_dir  = 0;      // searched file/dir type
    uint32_t   dentry;           // directory entry index 
    int32_t    found   = 0;      // not found (0) / name found (1) / end of dir (-1)

    // scan the parent directory mapper
    while ( found == 0 )
    {
        // get one page
        page = mapper_get_page( mapper , index );

        assert( (page != NULL) , __FUNCTION__ , "bad parent mapper\n");

        // get page base
        xptr_t base_xp = ppm_page2base( XPTR( local_cxy , page ) );
        base = (uint8_t *)GET_PTR( base_xp );

#if (DEBUG_FATFS_LOAD & 0x1)
if( DEBUG_FATFS_LOAD < cycle )
{
    uint32_t * buf = (uint32_t *)base;
    uint32_t line , word;
    printk("\n***** %s : First 16 dentries for parent inode %x\n",
    __FUNCTION__ , parent_inode );
    for( line = 0 ; line < 16 ; line++ )
    {
        printk("%X : ", line );
        for( word = 0 ; word < 8 ; word++ ) printk("%X ", buf[(line<<4) + word] );
        printk("\n");
    }
}
#endif
        // scan this page until end of directory, end of page, or name found
        while( (offset < 4096) && (found == 0) )
        {
            attr = fatfs_get_record( DIR_ATTR , base + offset , 0 );   
            ord  = fatfs_get_record( LDIR_ORD , base + offset , 0 );   

            if (ord == NO_MORE_ENTRY)                 // no more entry => break
            {
                found = -1;
            }
            else if ( ord == FREE_ENTRY )             // free entry => skip
            {
                offset = offset + 32;
            }
            else if ( attr == ATTR_LONG_NAME_MASK )   // LFN entry => get partial cname
            {
                seq = ord & 0x3;
                lfn = (seq > lfn) ? seq : lfn;   
                if      ( seq == 1 ) fatfs_get_name_from_long( base + offset, lfn1 );
                else if ( seq == 2 ) fatfs_get_name_from_long( base + offset, lfn2 );
                else if ( seq == 3 ) fatfs_get_name_from_long( base + offset, lfn3 );
                offset = offset + 32;
            }
            else                                 // NORMAL entry
            {
                // build the extracted name
                if      ( lfn == 0 )
                {
                    fatfs_get_name_from_short( base + offset , cname );
                }
                else if ( lfn == 1 )
                {
                    strcpy( cname      , lfn1 );
                }   
                else if ( lfn == 2 ) 
                {
                    strcpy( cname      , lfn1 );
                    strcpy( cname + 13 , lfn2 );
                }
                else if ( lfn == 3 ) 
                {
                    strcpy( cname      , lfn1 );
                    strcpy( cname + 13 , lfn2 );
                    strcpy( cname + 26 , lfn3 );
                }

                // get dentry arguments if extracted cname == searched name
                if ( strcmp( name , cname ) == 0 )
                {
                    cluster = (fatfs_get_record( DIR_FST_CLUS_HI , base + offset , 1 ) << 16) |
                              (fatfs_get_record( DIR_FST_CLUS_LO , base + offset , 1 )      ) ;
                    dentry  = ((index<<12) + offset)>>5;
                    is_dir  = ((attr & ATTR_DIRECTORY) == ATTR_DIRECTORY);
                    size    = fatfs_get_record( DIR_FILE_SIZE , base + offset , 1 );
                    found   = 1;
                }
                offset = offset + 32;
                lfn    = 0;
            }
        }  // end loop on directory entries
        index++;
        offset = 0;
    }  // end loop on pages

    // analyse the result of scan

    if ( found == -1 )  // found end of directory => failure 
    {

#if DEBUG_FATFS_LOAD
cycle = (uint32_t)hal_get_cycles();
if( DEBUG_FATFS_LOAD < cycle )
printk("\n[DBG] %s : thread %x exit / child <%s> not found / cycle %d\n",
__FUNCTION__ , CURRENT_THREAD, name, cycle );
#endif

        return ENOENT;
    }
    else               // found searched child name
    {
        // get child inode cluster and local pointer
        cxy_t         child_cxy = GET_CXY( child_inode_xp );
        vfs_inode_t * child_ptr = (vfs_inode_t *)GET_PTR( child_inode_xp );

        // update the child inode "type", "size", and "extend" fields
        vfs_inode_type_t type = (is_dir) ? INODE_TYPE_DIR : INODE_TYPE_FILE;

        hal_remote_sw( XPTR( child_cxy , &child_ptr->type   ) , type );
        hal_remote_sw( XPTR( child_cxy , &child_ptr->size   ) , size );
        hal_remote_sw( XPTR( child_cxy , &child_ptr->extend ) , cluster );

#if DEBUG_FATFS_LOAD
cycle = (uint32_t)hal_get_cycles();
if( DEBUG_FATFS_LOAD < cycle )
printk("\n[DBG] %s : thread %x exit / child <%s> loaded / cycle %d\n",
__FUNCTION__ , CURRENT_THREAD, name, cycle );
#endif

        return 0;
    }
}  // end fatfs_inode_load()
