/*
 * stdlib.h - User level library definition.
 * 
 * Author     Alain Greiner (2016,2017)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef _STDLIB_H_
#define _STDLIB_H

#define   RAND_MAX   65535
  
/*********************************************************************************************
 * This function tests the <expression> argument value. 
 * When the value is false, it displays an error message, and terminates the calling thread.
 *********************************************************************************************
 * @ expression : Boolean expression to be checked.
 ********************************************************************************************/
void assert( int expression ); 

/*********************************************************************************************
 * This function TODO
 *********************************************************************************************
 ********************************************************************************************/
int atoi(const char *str);

/*********************************************************************************************
 * This function TODO
 *********************************************************************************************
 ********************************************************************************************/
double atof(const char *str);

/*********************************************************************************************
 * This function copies the content of a <src> byte array to a <dst> byte array.
 * Behaviour is undefined if <src> and <dst> arrays overlap.
 *********************************************************************************************
 * @ dst     : pointer on destination array.
 * @ dst     : pointer on source array.
 * @ size    : number of bytes to move.
 * @ return first argument.
 ********************************************************************************************/
void * memcpy( void         * dst, 
               const void   * src,
               unsigned int   size );

/*********************************************************************************************
 * This function fill a byte array with a byte value.
 *********************************************************************************************
 * @ dst     : pointer on the byte array.
 * @ s       : byte value (will be casted to unsigned char).
 * @ size    : number of bytes to be set.
 * @ return first argument.
 ********************************************************************************************/
void * memset( void        * dst,
               int           s, 
               unsigned int  size);

/*********************************************************************************************
 * This function writes a formated string to the standard "stdout" stream.
 *********************************************************************************************
 * @ returns number of characters written if success / returns -1 if failure.
 ********************************************************************************************/
int printf( const char * format, ... );

/*********************************************************************************************
 * This function returns a positive integer fom the standard "stdin" stream.
 *********************************************************************************************
 * returns the integer value if success / returns -1 if failure.
 ********************************************************************************************/
int getint();

/*********************************************************************************************
 * This function writes one single character to the standard "stdout" stream.
 *********************************************************************************************
 * @ returns written character code if success / returns 0 (EOF) if failure.
 ********************************************************************************************/
int putchar( int c );

/*********************************************************************************************
 * This function returns one single character from the standard "stdin" stream.
 *********************************************************************************************
 * @ returns read character code if success / returns 0 (EOF) if failure.
 ********************************************************************************************/
int getchar();

/*********************************************************************************************
 * This function copies a formated string to a fixed size buffer. 
 * it includes the NUL terminating character. 
 * it cheks that the formated string fit in the buffer length.
 *********************************************************************************************
 * @ string    :  pointer on target buffer.
 * @ length    : max bumber of characters in target buffer.
 * @ format    : formated string.
 * @ returns number of characters written if success / returns -1 if failure.
 ********************************************************************************************/
int snprintf( char         * string,
              unsigned int   length,
              const char   * format, ... );

/*********************************************************************************************
 * This function sets the seed for a new sequence of pseudo-random numbers to be returned
 * by the rand function rand(). These sequences are repeatable by calling srand() with 
 * the same seed value.
 *********************************************************************************************
 * # seed  : seed value.
 ********************************************************************************************/
void srand( unsigned int seed );

/*********************************************************************************************
 * This function computes a sequence of pseudo-random integers in the range [0 to RAND_MAX].
 *********************************************************************************************
 * @ return an integer value between 0 and RAND_MAX.
 ********************************************************************************************/
int rand();

/*********************************************************************************************
 * This blocking function implements an user-level interactive debugger that can be
 * introduced in any user application to display various kernel distributed structures
 * related to the calling process. The supported commands are:
 * - p (cxy)     : display all processes descriptors in a given cluster.
 * - s (cxy,lid) : display all threads attached to a given core in a given cluster.
 * - v (cxy)     : display the calling process VMM in a given cluster.
 * - t (tid)     : display all owner process descriptors attached to a given TXT terminal.
 * - x           : force the calling process to exit.
 * - c           : continue calling process execution.
 *********************************************************************************************
 * @ return an integer value between 0 and RAND_MAX.
 ********************************************************************************************/
void idbg();


#endif  // _STDLIB_H_
