/*
 * remote_sem.c - Kernel function implementing the semaphore related syscalls.
 * 
 * Author   Alain Greiner  (2016)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_types.h>
#include <thread.h>
#include <kmem.h>
#include <printk.h>
#include <process.h>
#include <vmm.h>
#include <remote_sem.h>


///////////////////////////////////////////////
xptr_t remote_sem_from_vaddr( intptr_t  vaddr )
{
    // get pointer on local process_descriptor
    process_t * process = CURRENT_PROCESS;

    // get extended pointer on reference process
    xptr_t      ref_xp = process->ref_xp;

    // get cluster and local pointer on reference process 
    cxy_t          ref_cxy = GET_CXY( ref_xp );
    process_t    * ref_ptr = (process_t *)GET_PTR( ref_xp );

    // get extended pointer on root of semaphores list 
    xptr_t root_xp = XPTR( ref_cxy , &ref_ptr->sem_root );
   
    // scan reference process semaphores list
    xptr_t         iter_xp;
    xptr_t         sem_xp;
    cxy_t          sem_cxy;
    remote_sem_t * sem_ptr;
    intptr_t       ident;
    bool_t         found = false;
            
    XLIST_FOREACH( root_xp , iter_xp )
    {
        sem_xp  = XLIST_ELEMENT( iter_xp , remote_sem_t , sem_list );
        sem_cxy = GET_CXY( sem_xp );
        sem_ptr = (remote_sem_t *)GET_PTR( sem_xp );
        ident   = hal_remote_lw( XPTR( sem_cxy , &sem_ptr->ident ) );   
        if( ident == vaddr )
        {
            found = true;
            break;
        }
    }

    if( found == false )  return XPTR_NULL;
    else                  return sem_xp;

}  // end remote_sem_from_vaddr()

/////////////////////////////////////////
error_t remote_sem_init( intptr_t  vaddr,
                         uint32_t  value )
{
    xptr_t         sem_xp;
    remote_sem_t * sem_ptr;

    // get pointer on local process descriptor
    process_t * process = CURRENT_PROCESS;

    // get extended pointer on reference process
    xptr_t      ref_xp = process->ref_xp;

    cxy_t       ref_cxy = GET_CXY( ref_xp );
    process_t * ref_ptr = (process_t *)GET_PTR( ref_xp );

    // allocate memory for new semaphore in reference cluster
    if( ref_cxy == local_cxy )  // local cluster is the reference 
    {
        kmem_req_t req;   
        req.type  = KMEM_SEM;
        req.flags = AF_ZERO;
        sem_ptr   = kmem_alloc( &req );
        sem_xp    = XPTR( local_cxy , sem_ptr );
    }
    else                         // reference is remote
    {
        rpc_semaphore_alloc_client( ref_cxy , &sem_xp );
        sem_ptr = (remote_sem_t *)GET_PTR( sem_xp );
    }

    if( sem_xp == XPTR_NULL ) return ENOMEM;

    // initialise semaphore lock
    remote_spinlock_init( XPTR( ref_cxy , &sem_ptr->lock ) );

    // initialise semaphore count
    hal_remote_sw( XPTR( ref_cxy , &sem_ptr->count ) , value );

    // initialise vaddr
	hal_remote_spt( XPTR( ref_cxy , &sem_ptr->ident ) , (void *)vaddr );

    // initialise waiting threads queue
	xlist_root_init( XPTR( ref_cxy , &sem_ptr->wait_queue ) );

    // register new semaphore in reference process xlist
    xptr_t root_xp = XPTR( ref_cxy , &ref_ptr->sem_root );
    xptr_t xp_list = XPTR( ref_cxy , &sem_ptr->sem_list );
    xlist_add_first( root_xp , xp_list );

    return 0;

}  // en remote_sem_init()
 
//////////////////////////////////
void remote_sem_wait( xptr_t sem_xp )
{ 
    // get semaphore cluster and local pointer
    cxy_t          sem_cxy = GET_CXY( sem_xp );
    remote_sem_t * sem_ptr = (remote_sem_t *)GET_PTR( sem_xp );

    // get lock protecting semaphore     
	remote_spinlock_lock( XPTR( sem_cxy , &sem_ptr->lock ) );
  
    // get semaphore current value
    uint32_t count = hal_remote_lw( XPTR( sem_cxy , &sem_ptr->count ) );

	if( count > 0 )       // success
	{
        // decrement semaphore value
        hal_remote_sw( XPTR( sem_cxy , &sem_ptr->count ) , count - 1 );

        // release lock
	    remote_spinlock_unlock( XPTR( sem_cxy , &sem_ptr->lock ) );
	}
	else                 // failure
	{
        thread_t * this = CURRENT_THREAD;

        // register thread in waiting queue
        xptr_t root_xp   = (xptr_t)hal_remote_lwd( XPTR( sem_cxy , &sem_ptr->wait_queue ) );
        xptr_t thread_xp = XPTR( local_cxy , this );
		xlist_add_last( root_xp , thread_xp );

        // release lock
	    remote_spinlock_unlock( XPTR( sem_cxy , &sem_ptr->lock ) );

        // block and deschedule
        thread_block( this , THREAD_BLOCKED_SEM );  
        sched_yield();
	}
}  // end remote_sem_wait()

/////////////////////////////////////
void remote_sem_post( xptr_t sem_xp )
{
    // get semaphore cluster and local pointer
    cxy_t          sem_cxy = GET_CXY( sem_xp );
    remote_sem_t * sem_ptr = (remote_sem_t *)GET_PTR( sem_xp );

    // get lock protecting semaphore
	remote_spinlock_lock( XPTR( sem_cxy , &sem_ptr->lock ) );
  
    // get remote pointer on waiting queue root
    xptr_t queue_xp = (xptr_t)hal_remote_lwd( XPTR( sem_cxy , &sem_ptr->wait_queue ) );
  
	if( xlist_is_empty( queue_xp ) )   // no waiting thread
    {
        // get semaphore current value
        uint32_t count = hal_remote_lw( XPTR( sem_cxy , &sem_ptr->count ) );

        // increment semaphore value
        hal_remote_sw( XPTR( sem_cxy , &sem_ptr->count ) , count + 1 );
    }
    else
    {
        // get first waiting thread from queue
        xptr_t thread_xp = XLIST_FIRST_ELEMENT( queue_xp , thread_t , wait_list );

        // get thread cluster and local poiner
        cxy_t      thread_cxy = GET_CXY( thread_xp );
        thread_t * thread_ptr = (thread_t *)GET_PTR( thread_xp );

        // remove the thread from the waiting queue, and unblock
        xlist_unlink( XPTR( thread_cxy , &thread_ptr->wait_list ) );
		thread_unblock( thread_xp , THREAD_BLOCKED_SEM );
    }

    // release lock
	remote_spinlock_unlock( XPTR( sem_cxy , &sem_ptr->lock ) );

}  // end remote_sem_post()

////////////////////////////////////////
void remote_sem_destroy( xptr_t sem_xp )
{
    // get semaphore cluster and local pointer
    cxy_t          sem_cxy = GET_CXY( sem_xp );
    remote_sem_t * sem_ptr = (remote_sem_t *)GET_PTR( sem_xp );

    // get lock protecting semaphore
    remote_spinlock_lock( XPTR( sem_cxy , &sem_ptr->lock ) );
  
    // get remote pointer on waiting queue 
    xptr_t queue_xp = (xptr_t)hal_remote_lwd( XPTR( sem_cxy , &sem_ptr->wait_queue ) );
  
    if( !xlist_is_empty( queue_xp ) )   // user error
    {
        printk("WARNING in %s for thread %x in process %x : "
               "destroy semaphore, but  waiting threads queue not empty\n", 
               __FUNCTION__ , CURRENT_THREAD->trdid , CURRENT_PROCESS->pid );
    }

    // reset semaphore count
    hal_remote_sw( XPTR( sem_cxy , &sem_ptr->count ) , 0 );

    // remove semaphore from process
    xptr_t xp_list = (xptr_t)hal_remote_lwd( XPTR( sem_cxy , &sem_ptr->sem_list ) );
    xlist_unlink( xp_list );

    // release lock
    remote_spinlock_unlock( XPTR( sem_cxy , &sem_ptr->lock ) );

    // release memory allocated
    if( sem_cxy == local_cxy )         // reference is local
    {
        kmem_req_t  req;
        req.type = KMEM_SEM;
        req.ptr  = sem_ptr;
        kmem_free( &req );
    }
    else                                // reference is remote
    {
        rpc_semaphore_free_client( sem_cxy , sem_ptr );
    }

}  // end remote_sem_destroy()

//////////////////////////////////////////////
void remote_sem_get_value( xptr_t      sem_xp,
                           uint32_t  * data )
{
    // get semaphore cluster and local pointer
    cxy_t          sem_cxy = GET_CXY( sem_xp );
    remote_sem_t * sem_ptr = (remote_sem_t *)GET_PTR( sem_xp );

    *data = hal_remote_lw( XPTR( sem_cxy , &sem_ptr->count ) );

}  // end remote_sem_get_value()


