/*
 * cluster_info.h - An array in cluster descriptor describing the whole mesh
 * 
 * Authors  Nicolas Phan    (2018)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef _CLUSTER_INFO_H_
#define _CLUSTER_INFO_H_

#include <hal_kernel_types.h>
#include <kernel_config.h>
#include <hard_config.h>

/*
 * cluster_info[] is an array contained in each cluster descriptor and giving
 * information about all clusters. It enables each cluster to access various
 * information about all the other clusters.
 * 
 * This array is present in the boot_info structured and thus initialized
 * in boot_info_init(). Then it is copied from the boot_info structure to
 * each cluster descriptor (cluster_t structure) by each cluster.
 * 
 * Every entry of cluster_info[] is a bitfield giving various info about
 * the working state of the cluster, incating whether things work or not.
 * 
 * In the bitfield, LSBs 0 to 'NB_TOTAL_PROCS - 1' indicate if
 * the cores work or not.
 * The MSB indicates if the cluster is active (is not a hole in the mesh)
 * then the second MSB indicate if it's the IO cluster, etc.
 */

#define CINFO_ACTIVE    0x8000    // if the current cluster contains cores
#define CINFO_IS_IO     0x4000    // if the current cluster is the io cluster
/*
#define CINFO_RAM_OK    0x2000    // if the current cluster has working RAM
#define CINFO_XCU_OK    0x1000    // if the current cluster has working XCU
// A mask with the NB_TOTAL_PROCS least significant bits at 1
#define CINFO_CORES_MASK ((1 << NB_TOTAL_PROCS) - 1)
// This mask, when assigned to a cluster_info[] entry, indicates that the cluster
// is fully functionnal and all of its cores work
#define CINFO_CLUSTER_OK (CINFO_ACTIVE | CINFO_RAM_OK | CINFO_XCU_OK | CINFO_CORES_MASK)
*/

/******************************************************************************************
 * This function indicates if a given cluster is active (contains cores)
 ******************************************************************************************
 * @ cluster_info      : The cluster_info[] entry corresponding to the target cluster
 * @ return -1 if error, =0 if the cluster is empty or 1 if the cluster has at least 1 core
 *****************************************************************************************/
int cluster_info_is_active( uint32_t cluster_info );

/******************************************************************************************
 * This function gives the total number of active clusters (those who have cores, RAM...)
 ******************************************************************************************
 * @ cluster_info      : The cluster_info[] entry corresponding to the target cluster
 * @ return the number of active clusters
 *****************************************************************************************/
int cluster_info_nb_actives( uint32_t cluster_info[CONFIG_MAX_CLUSTERS_X][CONFIG_MAX_CLUSTERS_Y] );

/******************************************************************************************
 * This function indicates if a given cluster is active (has cores) and functionnal (works)
 ******************************************************************************************
 * @ cluster_info      : The cluster_info[] entry corresponding to the target cluster
 * @ return -1 if error, =0 if the cluster is faulty or >0 if the core is working
 *****************************************************************************************/
// int cluster_info_cluster_ok( uint16_t cluster_info );

/******************************************************************************************
 * This function indicates if a specific core is working in the given cluster
 ******************************************************************************************
 * @ cluster_info      : The cluster_info[] entry corresponding to the target cluster
 * @ n                 : The lid of the core of interest
 * @ return -1 if error, 0 if the core is faulty or >0 if the core is working
 *****************************************************************************************/
// int cluster_info_core_ok( uint16_t cluster_info, int n );

#endif